//
//  SRScheduleItem.swift
//  BandKit
//
//  Created by Mac on 2020/5/7.
//

import Foundation

public class SRScheduleItem: SRModelBase, SRModelProtocol {
    /// 开关: 0 关闭 、1 打开
    public var feature: UInt8
    /// 时间戳，是相对于1970-1-1的时间 单位 秒 (年/月/日 时：分)
    public var time: TimeInterval
    /// 震动
    public var vibrate: SRVibrate
    /// 闹钟标签：max ( 30 Byte )
    public var label: String
    
    public var date: Date {
        get {
            Date(timeIntervalSince1970: time)
        }
        set {
            time = newValue.timeIntervalSince1970
        }
    }
    
    public var datetime: String { return dformatter.string(from: Date(timeIntervalSince1970: time)) }

    ///: - Initialize
    required public init() {
        self.feature = 1
        self.time = Date().timeIntervalSince1970 + 120
        self.vibrate = SRVibrate()
        self.label = ""
    }
    
    
    ///: - internal
    
    var invalid: Bool {
        return (feature==0) && (time <= date.timeIntervalSince1970  && vibrate.byte==0 && label.count==0)
    }

    
    lazy var dformatter: DateFormatter = {
        let dformatter = DateFormatter.gregorian
        dformatter.dateFormat = "yyyy-MM-dd HH:mm"
        return dformatter
    }()
    
    var maxLabelLen: Int { 30 }


    ///: - override

    override var dataLen: Int { 40 }

    override func data() -> Data {
        let calendar = SRCalendarUtils.calendar
        let date = Date(timeIntervalSince1970: time)
        let unitFlags: Set<Calendar.Component> = [.year, .month, .day, .hour, .minute, .second, .weekday]
        let components =  calendar.dateComponents(unitFlags, from: date)
        
        let year    = UInt16(components.year!)
        let month   = UInt8(components.month!)
        let day     = UInt8(components.day!)
        let hour    = UInt8(components.hour!)
        let minute  = UInt8(components.minute!)
        let _ringSet: UInt8 = vibrate.byte
        var _labelLen = 0
        var _label = Data()
        if let hasData = label.data(using: .utf8) { _labelLen = min(hasData.count, maxLabelLen); _label += hasData.prefix(maxLabelLen) }
        if _label.count < maxLabelLen { _label += Data(count: maxLabelLen-_label.count) }

        
        let buffer = SRByteBuffer()
        buffer.putShort(year)
        buffer.putByte(feature)
        buffer.putByte(month)
        buffer.putByte(day)
        buffer.putByte(hour)
        buffer.putByte(minute)
        buffer.putByte(_ringSet)
        buffer.putByte(UInt8(_labelLen))
        buffer.putData(_label) // 30 Byte
        buffer.putByte(0x00) // 对齐

        return buffer.bufferData
    }
    
    
    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false}
        
        let buffer = SRByteBuffer(with: data)
        let _year    = buffer.getShort()
        let _feature  = buffer.get()
        let _month   = buffer.get()
        let _day     = buffer.get()
        let _hour    = buffer.get()
        let _minute  = buffer.get()
        let _ringSet   = buffer.get()
        let _lableLen = Int(buffer.get())
        let _lableData = buffer.getBytes(min(_lableLen, maxLabelLen))
        buffer.position = 39
        let _ = buffer.get()

        var components =  DateComponents()
        components.year     = Int(_year)
        components.month    = Int(_month)
        components.day      = Int(_day)
        components.hour     = Int(_hour)
        components.minute   = Int(_minute)
        let calendar = Calendar(identifier: .gregorian)
        let date = calendar.date(from: components)
        let _time = date?.timeIntervalSince1970 ?? 0
        
        self.feature = _feature
        self.time = _time
        self.vibrate.byte = _ringSet
        self.label = String(data: _lableData, encoding: String.Encoding.utf8) ?? ""
        return true
    }
}
