//
//  SRDetectItem.swift
//  BandKit
//
//  Created by Mac on 2020/5/11.
//

import Foundation

/// 检测的类型
public enum SRDetectType: Int, SREnumtProtocol {
    case bpm
    case bp
    var ext: UInt8 {
        switch self {
        case .bpm:   return 0x01
        case .bp:    return 0x02
        }
    }
    
    var value: UInt8 {
        switch self {
        case .bpm:   return 0x01
        case .bp:    return 0x02
        }
    }
    
    static let collection: [SRDetectType] = [.bpm, .bp]
    
    static func parse(_ value: UInt8) -> SRDetectType? {
        return collection.first(where: { $0.value == value } )
    }
    
}

extension SRDetectType: CustomStringConvertible {
    public var description: String {
        switch self {
        case .bpm:   return "bpm"
        case .bp:    return "bp"
        }
    }
}

/// 检测项
/// 分钟时间戳，高值 和 低值
public class SRDetectItem: SRModelBase, SRModelProtocol {
    /// 类型
    public private(set) var type: SRDetectType = .bpm
    /// 时间戳 (单位 秒 ) 相对于1970 的时间
    /// yyyy-MM-dd HH:mm
    public private(set) var time: TimeInterval = 0
    /// 心率 或者 血压高压值 (收缩压)
    public private(set) var heartorsbp: UInt8 = 0
    /// 血氧 或者 血压低压值 (舒张压)
    public private(set) var spo2ordbp: UInt8 = 0
    /// 结束检测
    public var end: Bool { return _end != nil }
    

    public var timeText: String { return dformatterTime.string(from: Date(timeIntervalSince1970: time)) }
    public var dateText: String { return dformatterDate.string(from: Date(timeIntervalSince1970: time)) }


    ///: - Initialize
    required public init() {}
    init(with record: DetectRecord) {
        type = SRDetectType.parse(record.type)!
        time = record.time
        heartorsbp = record.heartorsbp
        spo2ordbp = record.spo2ordbp
    }
    
    init(with t: SRDetectType) {
        type = t
    }
   
    ///: - internal
    
    var _end: String? = nil
    
    var dateTime: String { return dformatterDateTime.string(from: Date(timeIntervalSince1970: time)) }
    
    var valid: Bool {
        if type == .bpm  { return (time>0 && heartorsbp >= 40) }
        if type == .bp   { return (time>0 && heartorsbp >  spo2ordbp && spo2ordbp>20) }
        return false
    }
    
    lazy var dformatterDate: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "MM-dd"
           return dformatter
    }()

    lazy var dformatterTime: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "HH:mm"
           return dformatter
    }()

    lazy var dformatterDateTime: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "yyyy-MM-dd HH:mm"
           return dformatter
    }()
        
    var endLen: Int { 3 }

    ///: - override
    
    override var dataLen: Int { 8 }

    override func data() -> Data {
        let date = Date(timeIntervalSince1970: time)
        let calendar = Calendar(identifier: .gregorian)

        let unitFlags: Set<Calendar.Component> = [.year, .month, .day, .hour, .minute]
        let components =  calendar.dateComponents(unitFlags, from: date)
        
        let year    = UInt16(components.year!)
        let month   = UInt8(components.month!)
        let day     = UInt8(components.day!)
        let hour    = UInt8(components.hour!)
        let minute  = UInt8(components.minute!)

        
        let buffer = SRByteBuffer()
        buffer.putShort(year)
        buffer.putByte(month)
        buffer.putByte(day)
        buffer.putByte(hour)
        buffer.putByte(minute)
        buffer.putByte(heartorsbp)
        buffer.putByte(spo2ordbp)
        
        return buffer.bufferData
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        if data.count == endLen { _end = String(data: data, encoding: String.Encoding.utf8); return true}
        guard data.count == dataLen else { return false }
        
        let buffer = SRByteBuffer(with: data)
        let year    = buffer.getShort()
        let month   = buffer.get()
        let day     = buffer.get()
        let hour    = buffer.get()
        let minute  = buffer.get()
        heartorsbp = buffer.get()
        spo2ordbp  = buffer.get()

        var components =  DateComponents()
        components.year     = Int(year)
        components.month    = Int(month)
        components.day      = Int(day)
        components.hour     = Int(hour)
        components.minute   = Int(minute)
        
        let calendar = Calendar(identifier: .gregorian)
        let date = calendar.date(from: components)
        time = date?.timeIntervalSince1970 ?? 0
        
        return true
    }
}

// MARK: 数据库
extension Array where Element: SRDetectItem {
    @discardableResult
    func save() -> [SRDetectItem] {
        var result = [SRDetectItem]()
        forEach { (item) in if DetectRecord.save(with: item, old: true) { result.append(item) } }
        return result
    }
}

extension SRDetectItem {
    @discardableResult
    func save() -> [SRDetectItem] {
        var result = [SRDetectItem]()
        if DetectRecord.save(with: self, old: true) { result.append(self) }
        return result
    }
}


/// 限制最多读取 300 组
extension SRDetectItem {
    
    public static var heartRateItems = [SRDetectItem]()
    
    public static var bloodPressureItems = [SRDetectItem]()
    
    public static func readFromDB() {
        heartRateItems.removeAll()
        heartRateItems = DetectRecord.queryAllRecord(type: .bpm, limit: 300, old: true).map { (detectRecord) -> SRDetectItem in
            SRDetectItem(with: detectRecord)
        }
        bloodPressureItems.removeAll()
        bloodPressureItems = DetectRecord.queryAllRecord(type: .bp, limit: 300, old: true).map { (detectRecord) -> SRDetectItem in
            SRDetectItem(with: detectRecord)
        }
    }
}
