//
//  SRDeviceTime.swift
//  BandKit
//
//  Created by Mac on 2020/5/5.
//

import Foundation
import NEUtils

public class SRDeviceTime: SRModelBase, SRModelProtocol {
    /// 时间戳，是相对于1970-1-1的时间 单位 秒
    public private(set) var time: TimeInterval = Date().timeIntervalSince1970
    /// 天气类型： 0-12
    public private(set) var weatherType: NEWeatherType = .unknown
    /// 温度：温度必须发送摄氏度，[-50°, 50°]
    public private(set) var temperature: Int8 = 0
    
    ///: - Initialize
    public required init() { }
   
    ///: - internal
    var dateTime: String { return dformatter.string(from: Date(timeIntervalSince1970: time)) }

    lazy var dformatter: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "EEE, yyyy-MM-dd HH:mm:ss"
           return dformatter
    } ()

    ///: - override
    
    override var dataLen: Int { 10 }

    override func data() -> Data {
        let nowDate = Date()
        time = nowDate.timeIntervalSince1970
        let components =  SRCalendarUtils.components(ofDate: nowDate)
        
        let year    = UInt16(components.year!)
        let month   = UInt8(components.month!)
        let day     = UInt8(components.day!)
        let hour    = UInt8(components.hour!)
        let minute  = UInt8(components.minute!)
        let second  = UInt8(components.second!)
        let weekday = UInt8(components.weekday!) - 1

        let buffer = SRByteBuffer()
        buffer.putShort(year)
        buffer.putByte(month)
        buffer.putByte(day)
        buffer.putByte(hour)
        buffer.putByte(minute)
        buffer.putByte(second)
        buffer.putByte(weekday)
        
        if let currentHourItem = NEWeatherInfo.currentHourItem {
            weatherType = currentHourItem.type
            temperature = currentHourItem.temperature
        }
        
        buffer.putByte(UInt8(truncatingIfNeeded: weatherType.rawValue))
        buffer.putByte(UInt8(truncatingIfNeeded: temperature))
        
        return buffer.bufferData
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false}
        let buffer = SRByteBuffer(with: data)
        let year    = buffer.getShort()
        let month   = buffer.get()
        let day     = buffer.get()
        let hour    = buffer.get()
        let minute  = buffer.get()
        let second  = buffer.get()
        let _ = buffer.get()
        
        weatherType = NEWeatherType(rawValue: Int(buffer.get())) ?? .unknown
        temperature = Int8(buffer.get())

        let date = SRCalendarUtils.date(withYear: Int(year), month: Int(month), day: Int(day), hour: Int(hour), minute: Int(minute), second: Int(second))
        time = date.timeIntervalSince1970
        
        SRDeviceTime.readed = true
        return true
    }
}

extension SRDeviceTime: CustomStringConvertible {
    public var description: String {
        """
        {"time": \(dateTime), "weatherType": \(weatherType), "temperature": \(temperature),}
        """
    }
}


// MARK: - 数据缓存

extension SRDeviceTime {

    func saveToCache() {
        SRDeviceTime.default = self
    }
    
    static func readForCache() -> SRDeviceTime {
        if let cache = SRDeviceTime.default { return cache }
        return SRDeviceTime()
    }
    
    private static let key = "SRDeviceTimeBandKitKey"
    private static var _default: SRDeviceTime?
    private static var `default`: SRDeviceTime? {
        get {
            if let info = _default { return info}
            guard let jsonString = UserDefaults.standard.string(forKey: key) else { return nil }
            guard let info = SRDeviceTime.deserialize(from: jsonString) else { return nil }
            _default = info
            return _default
        }
        set {
            _default = newValue
            if _default != nil {
                UserDefaults.standard.set(_default!.jsonString, forKey: key)
            } else {
                UserDefaults.standard.removeObject(forKey: key)
            }
            UserDefaults.standard.synchronize()
        }
    }
}

/// 新连接设备是否读取过数据
extension SRDeviceTime {
    private static let readedKey = "SRDeviceTimeReadedKey"
    public static var readed: Bool {
        get {
            UserDefaults.standard.bool(forKey: readedKey)
        }
        set {
            UserDefaults.standard.set(newValue, forKey: readedKey)
            UserDefaults.standard.synchronize()
        }
    }
}
