//
//  SRDrinkInfo.swift
//  BandKit
//
//  Created by NEIL on 2020/5/10.
//

import Foundation

public enum SRGenderType: Int, SREnumtProtocol {
    case male
    case female
    var value: UInt8 {
        switch self {
        case .male: return 0x00
        case .female: return 0x01
        }
    }
    static func parse(_ value: UInt8) -> SRGenderType {
        if value == male.value { return.male }
        return .female
    }
}

extension SRGenderType: CustomStringConvertible {
    public var description: String {
        switch self {
        case .male: return "male"
        case .female: return "female"
        }
    }
}


public enum SRDateformatType: Int, SREnumtProtocol {
    case mmdd
    case ddmm
    var value: UInt8 {
        switch self {
        case .mmdd: return 0x00
        case .ddmm: return 0x01
        }
    }
    static func parse(_ value: UInt8) -> SRDateformatType {
        if value == mmdd.value { return.mmdd }
        return .ddmm
    }
}

extension SRDateformatType: CustomStringConvertible {
    public var description: String {
        switch self {
        case .mmdd: return "MM/DD"
        case .ddmm: return "DD/MM"
        }
    }
}

public class SRDrinkInfo: SRModelBase, SRModelProtocol {
    /// 喝水提醒
    public var waterRemind: SRRemindInterval = SRRemindInterval()
    /// 久坐提醒时间间隔 （分钟）, 默认 60
    public var sedentaryInterval: UInt8 = 60
    /// 日期显示格式:  0（ 默认值 — MMDD ）， 1 ... 其它
    public var dateFormat: SRDateformatType = .mmdd
    /// 性别
    public var gender: SRGenderType = .male
    /// 身高（cm）
    public var height: UInt8 = 180
    /// 血压检查参考值:  高压
    public var sbpReference: UInt8 = 110
    /// 血压检查参考值:  低压
    public var dbpReference: UInt8 = 90

    ///: - Initialize
    required public init() {}
    
    ///: - internal
    /// 保留字节
    var reserve: [UInt8] = [0, 0, 0, 0, 0]

    ///: - override
    
    override var dataLen: Int { 16 }

    override func data() -> Data {
        
        let buffer = SRByteBuffer()
        buffer.putData(waterRemind.data())
        buffer.putByte(sedentaryInterval)
        buffer.putByte(dateFormat.value)
        buffer.putByte(gender.value)
        buffer.putByte(height)
        buffer.putByte(sbpReference)
        buffer.putByte(dbpReference)
        buffer.putBytes(reserve)

        let _data = buffer.bufferData
        return _data
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false}

        let buffer = SRByteBuffer(with: data)
        waterRemind.parse(buffer.getBytes(5))
        sedentaryInterval = buffer.get()
        dateFormat = SRDateformatType.parse(buffer.get())
        gender = SRGenderType.parse(buffer.get())
        height = buffer.get()
        sbpReference = buffer.get()
        dbpReference = buffer.get()
        reserve = [UInt8](buffer.getBytes(5))
        
        SRDrinkInfo.readed = true
        return true
    }
}


// MARK: - 数据缓存

extension SRDrinkInfo {
    func saveToCache() {
        SRDrinkInfo.default = self
    }
    
    static func readForCache() -> SRDrinkInfo {
        if let cache = SRDrinkInfo.default { return cache }
        return SRDrinkInfo()
    }
    
    private static let key = "SRDrinkInfoBandKitKey"
    private static var _default: SRDrinkInfo?
    private static var `default`: SRDrinkInfo? {
        get {
            if let info = _default { return info}
            guard let jsonString = UserDefaults.standard.string(forKey: key) else { return nil }
            guard let info = SRDrinkInfo.deserialize(from: jsonString) else { return nil }
            _default = info
            return _default
        }
        set {
            _default = newValue
            if _default != nil {
                UserDefaults.standard.set(_default!.jsonString, forKey: key)
            } else {
                UserDefaults.standard.removeObject(forKey: key)
            }
            UserDefaults.standard.synchronize()
        }
    }
}

/// 新连接设备是否读取过数据
extension SRDrinkInfo {
    private static let readedKey = "SRDrinkInfoReadedKey"
    public static var readed: Bool {
        get {
            UserDefaults.standard.bool(forKey: readedKey)
        }
        set {
            UserDefaults.standard.set(newValue, forKey: readedKey)
            UserDefaults.standard.synchronize()
        }
    }
}

extension SRDrinkInfo {

    func csvContent() -> String {
        let titleItems: [String] = ["Sedentary interval", "Date format", "Gender", "Height", "Reference sbp", "Reference dbp"]
        let valueItems: [String] = ["\(sedentaryInterval)", "\(dateFormat)", "\(gender)", "\(height)", "\(sbpReference)", "\(dbpReference)"]
        var content = ""
        for i in 0..<titleItems.count {
            content += titleItems[i] + "," + valueItems[i] + "\n"
        }
        return waterRemind.csvContent(name: "Water remind") + content
    }
    
    public static func csvContent() -> String {
        return SRDataManager.drinkInfo.csvContent()
    }

}

