//
//  SRFirmwareInfo.swift
//  BandKit
//
//  Created by Mac on 2020/5/9.
//

import Foundation

public class SRFirmwareInfo: SRModelBase, SRModelProtocol {
    /// Model 号
    public private(set) var model: String = ""
    /// 版本号
    public private(set) var version: UInt8 = 0
    /// 功能号
    public private(set) var feature = SRFirmwareFeature()
    /// 显示版本
    public var showVersion: String { "V\(model.last ?? "0").1.\(version)"}
    
    ///: - Initialize
    required public init() {}
   
    ///: - internal
    /// 设备屏的信息, 0是没有屏, 1…N是后面规定
    /// 写非0值, 开始屏幕测试;整屏刷屏,测试屏幕是否有坏区
    var screen: UInt8 = 0
    /// 设备按键, 0是没有按键, 1…N是后面规定;
    /// 正常写0值, 非0值为删除bonding的功能;
    var key: UInt8 = 0
    /// 设备的字库IC类型,0是没有字库,1…N是后面规定
    /// 写非0值, 测试字库IC是否工作正常;
    var fontic: UInt8 = 0
    /// 设备的Gsensor类型,0是没有G sensor, 1…N是后面规定型号
    /// 写非0值, 测试Gsensor是否工作正常;
    var gsensor: UInt8 = 0
    /// 设备的马达类型, 0为没有马达, 1…N为后面规定的型号
    /// 写非0值, 测试马达转动是否正常
    var moto: UInt8 = 0
    /// 写设备的心率模块, 0是没有,1…N是后面规定型号
    /// 写非0值, 测试心率模块
    var heart: UInt8 = 0
    /// 设备的CFCA模块,0是没有, 1…N是后面规定型号
    /// 写非0值, 测试cfca模块工作是否正常
    var cfca: UInt8 = 0
    /// 设备的NFC模块, 0是没有, 1…N是后面规定型号
    /// 写非0值, 测试NFC模块工作是否正常
    var nfc: UInt8 = 0
    
    
    ///: - override
    
    override var dataLen: Int { 16 }

    override func data() -> Data {
        
        var _modelLen = 0
        var _realmodel: [UInt8] = [0, 0, 0, 0, 0, 0]
        if let _modelData = model.data(using: String.Encoding.utf8) {
            let _modelBytes = [UInt8](_modelData).prefix(_realmodel.count)
            _modelLen = _modelBytes.count
            _realmodel.replaceSubrange(_realmodel.count-_modelLen..<_realmodel.count, with: _modelBytes)
        }
        
        let buffer = SRByteBuffer()
        buffer.putByte(version)
        buffer.putByte(feature.feature)
        buffer.putByte(screen)
        buffer.putByte(key)
        buffer.putByte(fontic)
        buffer.putByte(gsensor)
        buffer.putByte(moto)
        buffer.putByte(heart)
        buffer.putByte(cfca)
        buffer.putByte(nfc)
        buffer.putBytes(_realmodel)

        let data = buffer.bufferData
        return data
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false}
        let buffer = SRByteBuffer(with: data)
        version = buffer.get()
        feature.feature = buffer.get()
        screen = buffer.get()
        key = buffer.get()
        fontic = buffer.get()
        gsensor = buffer.get()
        moto = buffer.get()
        heart = buffer.get()
        cfca = buffer.get()
        nfc = buffer.get()
        let modelBytes = [UInt8](buffer.getBytes(buffer.remain))

        var n = 0
        for e in modelBytes { if e != 0 { break }; n += 1 }
        
        let modelData = Data(modelBytes.suffix(modelBytes.count-n))
        model = String(data: modelData, encoding: String.Encoding.utf8) ?? ""
        
        SRFirmwareInfo.readed = true

        return true
    }
}

extension SRFirmwareInfo: CustomStringConvertible {
    public var description: String {
        """
        {"model": \(model), "version": \(version), "feature": \(feature)}
        """
    }
}





// MARK: - 数据缓存

extension SRFirmwareInfo {
    func saveToCache() {
        SRFirmwareInfo.default = self
    }
    
    static func readForCache() -> SRFirmwareInfo {
        if let cache = SRFirmwareInfo.default { return cache }
        return SRFirmwareInfo()
    }
    
    private static let key = "SRFirmwareInfoBandKitKey"
    private static var _default: SRFirmwareInfo?
    private static var `default`: SRFirmwareInfo? {
        get {
            if let info = _default { return info}
            guard let jsonString = UserDefaults.standard.string(forKey: key) else { return nil }
            guard let info = SRFirmwareInfo.deserialize(from: jsonString) else { return nil }
            _default = info
            return _default
        }
        set {
            _default = newValue
            if _default != nil {
                UserDefaults.standard.set(_default!.jsonString, forKey: key)
            } else {
                UserDefaults.standard.removeObject(forKey: key)
            }
            UserDefaults.standard.synchronize()
        }
    }
}

/// 新连接设备是否读取过数据
extension SRFirmwareInfo {
    private static let readedKey = "SRFirmwareInfoReadedKey"
    public static var readed: Bool {
        get {
            UserDefaults.standard.bool(forKey: readedKey)
        }
        set {
            UserDefaults.standard.set(newValue, forKey: readedKey)
            UserDefaults.standard.synchronize()
        }
    }
}
