//
//  SRExercise.swift
//  BandKit
//
//  Created by Mac on 2020/5/13.
//

import Foundation

// MARK: -  运动类型
public enum SRSportType: Int, SREnumtProtocol {
    case normal     // 普通计步模式
    case run        // 跑步
    case climbing   // 爬山
    case indoor     // 室内运动
    case swiming    // 游泳
    case yoga       // 瑜伽

    case walk       // 健步
    case badminton  // 羽毛球
    case basketball // 篮球
    case football   // 足球
    case pingPong   // 乒乓球
    case tennis     // 网球
    case bike       // 骑行
    case indoorRun  // 室内跑
    
    public var value: UInt8 {
        switch self {
        case .normal:       return 0x01
        case .run:          return 0x07
        case .climbing:     return 0x09
        case .indoor:       return 0x0A
        case .swiming:      return 0x0B
        case .yoga:         return 0x0C

        case .walk:         return 0x77
        case .badminton:    return 0x80
        case .basketball:   return 0x81
        case .football:     return 0x82
        case .pingPong:     return 0x85
        case .tennis:       return 0x87
        case .bike:         return 0x88
        case .indoorRun:    return 0x8A
        }
    }
    
    static var collection: [SRSportType] = [.normal, .run, .climbing, .indoor, swiming, .yoga, .walk,
                                          .badminton, .basketball, .football, .pingPong, .tennis, bike, .indoorRun]

    public static func parse(_ value: UInt8) -> SRSportType? {
        var type: SRSportType? = nil
        for e in collection { if e.value==value { type = e; break }}
        return type
    }

}

/// 一天的运动锻炼数据  ( 最多 10 组 )
public class SRExercise: SRModelBase, SRModelProtocol {
    // 日期
    public private(set) var date = SRFlashDateItem()
    /// 一天的运动锻炼数据   ( 最多 10 组 )
    public private(set) var records = [SRExerciseItem]()
    /// 总运动分钟数
    public private(set) var totalMinutes: Int = 0
    
    ///: - Initialize
    required public init() {}

    
    ///: - internal
    /// ( 0 ~ 10  )
    var count: UInt32 = 0
    var maxCount: Int { 10 }

    ///: - override
    
    override var dataLen: Int { 208 }

    override func data() -> Data {
        let buffer = SRByteBuffer()
        buffer.putData(date.data())
        buffer.putInt32(count)
        records.forEach { buffer.putData($0.data()) }
        return buffer.bufferData
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count >= dataLen else { return false}
        let buffer = SRByteBuffer(with: data)
        date.parse(buffer.getBytes(4))
        count = min(buffer.getInt32(), UInt32(maxCount))
        
        totalMinutes = 0
        records.removeAll()
        for i in 0..<count {
            let startIndex = i*20+8
            let endIndex = startIndex+19
            let itemData = data[startIndex...endIndex]
            let item = SRExerciseItem()
            item.parse(itemData)
            records.append(item)
            totalMinutes += Int(item.duration)
        }
        
        guard data.count >= 208 + totalMinutes else { return false}

        var index = 208
        for item in records {
            let startIndex = index
            let endIndex = startIndex + Int(item.duration) - 1
            let itemData = data[startIndex...endIndex]
            item.parseHeartRecords(itemData)
            index += Int(item.duration)
        }
        return true
    }
}

import HandyJSON
extension SRExercise {
    public func mapping(mapper: HelpingMapper) {
        mapper <<<
        self.date <-- TransformOf<SRFlashDateItem, String>(fromJSON: { (rawString) -> (SRFlashDateItem) in
            return SRFlashDateItem(dateStr: rawString ?? "")
        }, toJSON: { (dt) -> String? in
            return dt?.dateStr
        })
    }
}



// MARK: - 数据库查询
extension SRExercise {
    
    static func queryToday() -> SRExercise {
        let total = SRExercise()
        let dateStr = total.date.dateStr
        if let data = BandRecord.queryColumn(dateStr, .exercise) as? Data {
            total.parse(data)
        }
        return total
    }
    
    public static func query(with date: Date) -> SRExercise? {
        let dateStr = BandRecord.dateStr(date)
        if let data = BandRecord.queryColumn(dateStr, .exercise) as? Data {
            let total = SRExercise()
            total.parse(data)
            return total
        }
        return nil
    }

    public static func query(from fromDate: Date, to toDate: Date) -> [SRExercise] {
        var items = [SRExercise]()
        let fromDateStr = BandRecord.dateStr(fromDate)
        let toDateStr = BandRecord.dateStr(toDate)
        let records = BandRecord.queryRecord(from: fromDateStr, to: toDateStr, column: .exercise) as? [Data]
        records?.forEach({ (data) in
            let item = SRExercise()
            item.parse(data)
            items.append(item)
        })
        return items
    }
    
    public static func queryAll() -> [SRExercise] {
        var items = [SRExercise]()
        let records = BandRecord.queryAllRecord(column: .exercise) as? [Data]
        records?.forEach({ (data) in
            let item = SRExercise()
            item.parse(data)
            items.append(item)
        })
        return items
    }
}
