//
//  SRExerciseItem.swift
//  BandKit
//
//  Created by Mac on 2020/5/14.
//

import Foundation

/// 运动锻炼数据项
public class SRExerciseItem: SRModelBase, SRModelProtocol {
    // 运动类型
    public private(set) var type: SRSportType = .normal
    /// 持续时间 ( 分钟 )
    public private(set) var duration: UInt16 = 0
    /// 开始小时 （0~23）
    public private(set) var startHour: UInt8  = 0
    /// 开始分钟 （0~59）
    public private(set) var startMinute: UInt8  = 0
    /// 结束小时 （0~23）
    public private(set) var endHour: UInt8  = 0
    /// 结束分钟 （0~59）
    public private(set) var endMinute: UInt8  = 0
    /// 步数
    public private(set) var steps: UInt32  = 0
    /// 卡路里 （大卡）
    public private(set) var calories: UInt16  = 0
    /// 距离 （10 m）
    public private(set) var distance: UInt16  = 0
    /// 最大步频 ( 每分钟的步数 )
    public private(set) var maxSpace: UInt8  = 0
    /// 平均步频 ( 每分钟的步数 )
    public private(set) var avgSpace: UInt8  = 0
    /// 最大心率
    public private(set) var maxHeart: UInt8  = 0
    /// 平均心率
    public private(set) var avgHeart: UInt8  = 0
    /// 运动心率记录 （每分钟一次）
    public private(set) var heartRecords = [UInt8]()
    
    /// Period of time
    public var periodTime: String {
        String(format: "%02d:%02d~%02d:%02d", min(startHour, 23), min(startMinute, 59), min(endHour, 23), min(endMinute, 59))
    }
    
    ///: - Initialize
    required public init() {}

    
    ///: - override
    
    override var dataLen: Int { 20 }

    override func data() -> Data {
        let buffer = SRByteBuffer()
        buffer.putByte(type.value)
        buffer.putShort(duration)

        buffer.putByte(startHour)
        buffer.putByte(startMinute)
        buffer.putByte(endHour)
        buffer.putByte(endMinute)

        buffer.putInt32(steps)
        buffer.putShort(distance)
        buffer.putShort(calories)
        
        buffer.putByte(maxSpace)
        buffer.putByte(avgSpace)
        buffer.putByte(maxHeart)
        buffer.putByte(avgHeart)

        buffer.putBytes(heartRecords)
        
        return buffer.bufferData
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count >= dataLen else { return false}
        let buffer = SRByteBuffer(with: data)
        
        let typeValue = UInt8(truncatingIfNeeded: buffer.getShort())
        type = SRSportType.parse(typeValue) ?? .normal
        duration = buffer.getShort()
        
        startHour = buffer.get()
        startMinute = buffer.get()
        endHour = buffer.get()
        endMinute = buffer.get()
        
        steps = buffer.getInt32()
        distance = buffer.getShort()
        calories = buffer.getShort()
        
        maxSpace = buffer.get()
        avgSpace = buffer.get()
        maxHeart = buffer.get()
        avgHeart = buffer.get()

        return true
    }
    
    func parseHeartRecords(_ data: Data) {
        guard data.count == duration else { return }
        heartRecords = [UInt8](data)
    }
}

import HandyJSON
extension SRExerciseItem {
    public var description: String {
        """
        {"type":\(type),"duration":\(duration),periodTime":\(periodTime),"steps":\(steps),"calories":\(calories),"distance":\(distance),"maxSpace":\(avgSpace),"avgSpace":\(avgSpace),"maxHeart":\(maxHeart),"avgHeart":\(avgHeart)}
        """
    }
    
    public func mapping(mapper: HelpingMapper) {
        mapper >>> self.heartRecords
    }
}


extension SRExerciseItem {
    public static var debug: SRExerciseItem {
        let model = SRExerciseItem()
        // 时间
        model.duration = 40;
        model.startHour = 19;
        model.startMinute = 30;
        model.endHour = 20;
        model.endMinute = 10;
        // Steps 步数
        model.steps = 1234;
        model.distance = 250;
        model.calories = 345;
        // SF 步频
        model.avgSpace = 75;
        model.maxSpace = 88;
        // HR 心率
        model.avgHeart = 92;
        model.maxHeart = 136;
        
        for _ in 0..<40 {
            model.heartRecords.append(UInt8(random(min: 80, max: 140)))
        }
        return model
    }
}
