//
//  SRHeart.swift
//  BandKit
//
//  Created by Mac on 2020/5/13.
//

import Foundation

/// 一天的静态心率数据  ( 每10分钟记录一次，一天有144 次 )
public class SRHeart: SRModelBase, SRModelProtocol {
    // 日期
    public private(set) var date = SRFlashDateItem()
    /// 最高
    public private(set) var highest: UInt8 = 0
    /// 最底
    public private(set) var lowest: UInt8 = 0
    /// 平均
    public private(set) var aver: UInt8 = 0
    /// 一天的静态心率数据  ( 每10分钟记录一次，一天有144 次 )
    public private(set) var records = [UInt8]()
    /// 最近的
    public private(set) var latest: UInt8 = 0
    
    ///: - Initialize
    required public init() {}
    
    ///: - internal

    var blank: UInt8 = 0
    
    func loadDetailSleep() {
        /// details
        let nonzeroValues = records.filter({ $0 != 0 })
        guard let last = nonzeroValues.last else { return }
        latest = last
        lowest = last
        highest = last
        var total: UInt32 = 0
        for value in nonzeroValues {
            total += UInt32(value)
            lowest = min(lowest, value)
            highest = max(highest, value)
        }
        aver = UInt8(truncatingIfNeeded: total/UInt32(nonzeroValues.count))
    }

    
    ///: - override
    
    override var dataLen: Int { 152 }

    override func data() -> Data {
        let buffer = SRByteBuffer()
        buffer.putData(date.data())
        buffer.putByte(blank)
        buffer.putByte(highest)
        buffer.putByte(lowest)
        buffer.putByte(aver)
        buffer.putBytes(records)
        return buffer.bufferData
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false}
        let buffer = SRByteBuffer(with: data)
        date.parse(buffer.getBytes(4))
        blank = buffer.get()
        highest = buffer.get()
        lowest = buffer.get()
        aver = buffer.get()
        records = [UInt8](buffer.getBytes(buffer.remain))
        
        let nonzeroRecords = records.filter({ $0 != 0 })
        latest = nonzeroRecords.last ?? 0

        return true
    }
}

import HandyJSON
extension SRHeart: CustomStringConvertible {
    public var description: String {
        """
        {"date":\(date),"latest":\(latest),"highest":\(highest),"lowest":\(lowest),"lowest":\(lowest)}
        """
    }
    
    public func mapping(mapper: HelpingMapper) {
        mapper >>> self.records
        mapper >>> self.blank
    }
}



// MARK: - CSV 文件记录文本
extension SRHeart {
public static func csvTitle() -> String {
    let titleItems: [String] = ["Date", "Blanks", "Highest", "Lowest", "Average", "Records data (144)"]
    return titleItems.joined(separator: ",")
}

public func csvText() -> String {
    let recordText = records.map{String($0)}.joined(separator: " ")
    let text: [String] = ["\(date)", "\(blank)", "\(highest)", "\(lowest)", "\(aver)", recordText]
    return text.joined(separator: ",")
}
}

extension SRHeart {

    public static var debug: SRHeart {
        let model = SRHeart()
        
        var allValue: UInt64 = 0,  nonzeroCount: UInt64 = 0
        model.lowest = 200
        for _ in 0..<144 {
            let value = random(min: 65, max: 88)
            model.records.append(UInt8(value))
            model.lowest = UInt8(min(value, Int(model.lowest)))
            model.highest = UInt8(max(value, Int(model.highest)))
            if value != 0 { nonzeroCount += 1; allValue += UInt64(value) }
        }
        if nonzeroCount > 0 { model.aver = UInt8(allValue/nonzeroCount) }
        
        for enumerated in model.records.reversed().enumerated() {
            if enumerated.element > 0 { model.latest = enumerated.element; break  }
        }
        
        return model
    }
}


// MARK: - 数据库查询
extension SRHeart {
    static func queryToday() -> SRHeart {
        let total = SRHeart()
        let dateStr = total.date.dateStr
        if let data = BandRecord.queryColumn(dateStr, .heart) as? Data {
            total.parse(data)
        }
        return total
    }
    
    static func query(from fromDate: Date, to toDate: Date) -> [SRHeart] {
        var items = [SRHeart]()
        let fromDateStr = BandRecord.dateStr(fromDate)
        let toDateStr = BandRecord.dateStr(toDate)
        let records = BandRecord.queryRecord(from: fromDateStr, to: toDateStr, column: .heart) as? [Data]
        records?.forEach({ (data) in
            let item = SRHeart()
            item.parse(data)
            items.append(item)
        })
        return items
    }
    
    public static func query(details date: Date) -> SRHeart? {
        let dateStr = BandRecord.dateStr(date)
        if let data = BandRecord.queryColumn(dateStr, .heart) as? Data {
            let model = SRHeart()
            model.parse(data)
            return model
        }
        return nil
    }
    
    public static func query(sleep date: Date) -> SRHeart? {
        guard let date_yesterday = SRCalendarUtils.calendar.date(byAdding: .day, value: -1, to: date) else { return nil}
        let heart_yesterday = query(details: date_yesterday)
        let heart_today = query(details: date)
        var heartRecords: [UInt8]?
        if let beforeRecords = heart_yesterday?.records.suffix(24) {
            if heartRecords == nil { heartRecords = [UInt8]() }
            heartRecords? += beforeRecords
        }
        if let currentRecords = heart_today?.records.prefix(72) {
            if heartRecords == nil { heartRecords = [UInt8](repeating: 0, count: 24) }
            heartRecords? += currentRecords
        }
        guard let values = heartRecords else { return nil}
        let m = SRHeart()
        m.date = SRFlashDateItem(date: date)
        m.records = values
        m.loadDetailSleep()
        return m
    }
}
