//
//  SRSleep.swift
//  BandKit
//
//  Created by Mac on 2020/5/13.
//

import Foundation
import HandyJSON

public enum SRSleepType: Int, SREnumtProtocol {
    case none 
    case deep
    case light
    case wake
    
    public var value: UInt8 {
        switch self {
        case .none: return 0
        case .deep: return 1
        case .light: return 2
        case .wake: return 3
        }
    }
    
    var hkvalue: Int {
        switch self {
        case .none: return 0
        case .deep, .light: return 1
        case .wake: return 2
        }
    }
    
    static var collection: [SRSleepType] = [.none, .deep, .light, .wake]
    
    /// 1 Byte 存储值解出 1个记录
    public static func parse(_ value: UInt8) -> SRSleepType {
        var result: SRSleepType = .none
        for e in collection { if e.value == value { result = e; break } }
        return result
    }
    
    /// 1 Byte 存储值解出 4个记录
    static func parse(_ value: UInt8) -> [SRSleepType] {
        let records:[UInt8] = [(value) & 0x03,
                               (value >> 2) & 0x03,
                               (value >> 4) & 0x03,
                               (value >> 6) & 0x03]
        var result = [SRSleepType]()
        for e in records { result.append(parse(e)) }
        return result
    }
}

/// 一天的睡眠数据  ( 昨天 20:00 ~ 今天 12:00,  每分钟记录一次，总共 960 次 )
public class SRSleep: SRModelBase, SRModelProtocol {
    // 日期
    public private(set) var date = SRFlashDateItem()
    /// 总时间长 （ 分钟 ）
    public private(set) var totalMinutes: UInt16 = 0
    /// 深睡时长（ 分钟 ）
    public private(set) var deeplMinutes: UInt16 = 0
    /// 浅睡时长（ 分钟 ）
    public private(set) var lightMinutes: UInt16 = 0
    /// 清醒时长（ 分钟 ）
    public private(set) var wakeMinutes: UInt16 = 0

    /// 开始小时 （0~23）
    public private(set) var startHour: UInt8 = 0
    /// 开始分钟 （0~59）
    public private(set) var startMinute: UInt8 = 0
    /// 结束小时 （0~23）
    public private(set) var endHour: UInt8 = 0
    /// 结束分钟 （0~59）
    public private(set) var endMinute: UInt8 = 0
    /// 一天的睡眠数据  ( 昨天 20:00 ~ 今天 12:00,  每分钟记录一次，总共 960 次 )
    public private(set) var records = [SRSleepType]()
    
    /// 每小时的睡眠时间数组 (一共16个小时))
    public var hourRecords: [UInt8] {
        var result = [UInt8]()
        guard records.count == 960 else { return result}
        
        for i in 0..<16 {
            let startIndex = i*60
            let endIndex = startIndex+59
            let items = records[startIndex...endIndex].filter { (e) -> Bool in e != .none }
            result.append(UInt8(items.count))
        }
        return result
    }
    
    /// Period of time
    public var periodTime: String {
        String(format: "%02d:%02d-%02d:%02d", min(startHour, 23), min(startMinute, 59), min(endHour, 23), min(endMinute, 59))
    }
    
    ///: - Initialize
    required public init() {}

    ///: - internal
    /// 长度 240，每字节 4 分钟数据
    var collection: [UInt8] = [UInt8]()

    
    ///: - override
    
    override var dataLen: Int { 256 }

    override func data() -> Data {
        let buffer = SRByteBuffer()
        buffer.putData(date.data())
        buffer.putShort(totalMinutes)
        buffer.putShort(deeplMinutes)
        buffer.putShort(lightMinutes)
        buffer.putShort(wakeMinutes)
        buffer.putByte(startHour)
        buffer.putByte(startMinute)
        buffer.putByte(endHour)
        buffer.putByte(endMinute)
        buffer.putBytes(collection)
        return buffer.bufferData
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false}
        let buffer = SRByteBuffer(with: data)
        
        date.parse(buffer.getBytes(4))
        totalMinutes = buffer.getShort()
        deeplMinutes = buffer.getShort()
        lightMinutes = buffer.getShort()
        wakeMinutes  = buffer.getShort()
        
        startHour   = buffer.get()
        startMinute = buffer.get()
        endHour     = buffer.get()
        endMinute   = buffer.get()

        collection = [UInt8](buffer.getBytes(buffer.remain))
        
        records.removeAll()
        collection.forEach { (e) in records += SRSleepType.parse(e)}

        return true
    }
}

extension SRSleep: CustomStringConvertible {
    public var description: String {
        """
        {"date":\(date),"periodTime":\(periodTime),
        "totalMinutes":\(totalMinutes),"deeplMinutes":\(deeplMinutes),"lightMinutes":\(lightMinutes),"wakeMinutes":\(wakeMinutes)}
        """
    }
    
    public func mapping(mapper: HelpingMapper) {
        mapper >>> self.records
        mapper >>> self.collection
    }
}


// MARK: - CSV 文件记录文本
extension SRSleep {
    public static func csvTitle() -> String {
        let titleItems: [String] = ["Date", "Period time", "Total minutes", "Deepl minutes", "Light minutes", "Wake minutes", "Records data (240)"]
        return titleItems.joined(separator: ",")
    }
    
    public func csvText() -> String {
        let recordText = collection.map{String($0)}.joined(separator: " ")
        let text: [String] = ["\(date)", "\(periodTime)", "\(totalMinutes)", "\(deeplMinutes)", "\(lightMinutes)", "\(wakeMinutes)", recordText]
        return text.joined(separator: ",")
    }
}


extension SRSleep {
    
    public static var debug: SRSleep {
        let model = SRSleep()
        model.totalMinutes = 480
        model.deeplMinutes = 180
        model.lightMinutes = 240
        model.wakeMinutes = 60
        
        model.startHour = 0
        model.startMinute = 0
        model.endHour = 8
        model.endMinute = 0

        for index in 0..<960 {
            var type = SRSleepType.none
            if (index > 240 && index < 720) {
                if (index < 360) {
                    type = .deep
                } else if (index > 480 && index < 540) {
                    type = .deep
                } else if (index > 660) {
                    type = .wake
                } else {
                    type = .light
                }
            }
            model.records.append(type)
        }
        return model
    }
}

// MARK: - 数据库查询
extension SRSleep {
    
    static func queryToday() -> SRSleep {
        let total = SRSleep()
        let dateStr = total.date.dateStr
        if let data = BandRecord.queryColumn(dateStr, .sleep) as? Data {
            total.parse(data)
        }
        return total
    }
    
    public static func query(with date: Date) -> SRSleep? {
        let dateStr = BandRecord.dateStr(date)
        if let data = BandRecord.queryColumn(dateStr, .sleep) as? Data {
            let total = SRSleep()
            total.parse(data)
            return total
        }
        return nil
    }

    public static func query(from fromDate: Date, to toDate: Date) -> [SRSleep] {
        var items = [SRSleep]()
        let fromDateStr = BandRecord.dateStr(fromDate)
        let toDateStr = BandRecord.dateStr(toDate)
        let records = BandRecord.queryRecord(from: fromDateStr, to: toDateStr, column: .sleep) as? [Data]
        records?.forEach({ (data) in
            let item = SRSleep()
            item.parse(data)
            items.append(item)
        })
        return items
    }
}
