//
//  SRStep.swift
//  BandKit
//
//  Created by Mac on 2020/5/13.
//

import Foundation

/// 一天的计步数据 (每分钟记录一次，一天有1440次 )

public class SRStep: SRModelBase, SRModelProtocol {
    // 日期
    public private(set) var date = SRFlashDateItem()
    /// 总步数
    public private(set) var total: UInt32 = 0
    /// 每小时的计步数据记录 ( 24 小时 )
    public var hourRecords: [UInt16] = [UInt16](repeating: 0, count: 24)
    
    
    ///: - Initialize
    required public init() {}
    
    ///: - internal
    /// 一天的计步数据 (每分钟记录一次，一天有1440次 )
    private(set) var records = [UInt8]()
    /// 分段的记录，连续步数
    private(set) var sectioRecords = [SRStepItem]()

    
    ///: - override
    
    override var dataLen: Int { 1444 }

    override func data() -> Data {
        let buffer = SRByteBuffer()
        buffer.putData(date.data())
        buffer.putBytes(records)
        return buffer.bufferData
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false}
        let buffer = SRByteBuffer(with: data)
        
        date.parse(buffer.getBytes(4))
        records = [UInt8](buffer.getBytes(buffer.remain))
        parseHourRecords()
        parseSectionRecords()

        return true
    }
 
    private func parseHourRecords() {
        total = 0
        for i in 0..<hourRecords.count {
            var anHourSteps: UInt16 = 0
            for j in 0..<60 {
                anHourSteps += UInt16(records[i*60+j])
            }
            hourRecords[i] = anHourSteps
            total += UInt32(anHourSteps)
        }
    }
    
    private func parseSectionRecords() {
        sectioRecords.removeAll()
        let count = records.count
        var i = 0;
        while (i < count) {
            if (records[i] > 0) {
                // 统计非0段开始
                var sum: Int = Int(records[i]);
                var j = i+1;
                while j < count, records[j] > 0  {
                    sum += Int(records[j]);
                    j += 1;
                }
                let range = NSRange(location: i, length: j-i)
                if let item = SRStepItem.instance(sum, range) {
                    sectioRecords.append(item)
                }
                i = j+1
            } else {
                i += 1
            }
        }
    }
}

import HandyJSON
extension SRStep: CustomStringConvertible {
    public var description: String {
        """
        {"date": \(date), "total": \(total), "hourRecords": \(hourRecords)}
        """
    }
    
    public func mapping(mapper: HelpingMapper) {
        mapper >>> self.records
        mapper >>> self.sectioRecords
    }
}


// MARK: - 计算
extension SRStep {
    /// 距离 - 单位 m
    func distanceHours() -> [UInt16]{
        var array: [UInt16] = [UInt16](repeating: 0, count: 24);
        for i in 0..<array.count {
            var anHourDistance: Double = 0
            for j in 0..<60 {
                let anMinuteSteps = UInt16(records[i*60+j])
                anHourDistance += SRStep.distance(forMinute: anMinuteSteps)
            }
            array[i] = UInt16(anHourDistance)
        }
        return array;
    }

    /// 卡路里 - 单位大卡
    static func calorieHours(for distance: [UInt16]) -> [Double] {
        return distance.map { self.calorie(forHour: $0) }
    }
    
    
    private static let kMinRunningFrequency = 120; // 跑步最低步频
    private static let kActiveCalorieConstant = 1.036; // 卡路里参数
    
    /// 步数转距离 （步数 -> 米）
    private static func distance(forMinute steps: UInt16) -> Double {
        let user = SRUserInfo.readForCache().user
        if (steps >= kMinRunningFrequency) {
            return Double(steps) * Double(user.strideRun) / 100;
        } else {
            return Double(steps) * Double(user.strideWalk) / 100;
        }
    }
    
    /// 距离转卡路里 （米 -> 大卡）
    private static func calorie(forHour distance: UInt16) -> Double {
        let user = SRUserInfo.readForCache().user
        return (Double(user.weight) * Double(distance) * kActiveCalorieConstant)/1000;
    }
}

// MARK: - CSV 文件记录文本
extension SRStep {
    public static func csvTitle() -> String {
        let titleItems: [String] = ["Date", "Records data (24)"]
        return titleItems.joined(separator: ",")
    }
    
    public func csvText() -> String {
        let recordText = hourRecords.map{String($0)}.joined(separator: " ")
        let text: [String] = ["\(date)", recordText]
        return text.joined(separator: ",")
    }
}


extension SRStep {
    public static func debug(with totalStep: UInt32 = 8000, hours: Int=24) -> SRStep {
        let model = SRStep()
        model.total = totalStep
        
        var total: UInt32 = 0
        for idx in 0..<hours {
            var step: UInt16 = 0
            if total == totalStep { break }
            if totalStep - total < 1000 || idx == hours-1 {
                step = UInt16(totalStep - total) }
            else {
                step = UInt16(random(min: 15, max: 5000))
                if total + UInt32(step) > totalStep {
                    step = UInt16(totalStep - total)
                }
            }
            model.hourRecords[idx] = step
            total += UInt32(step)
        }
        return  model
    }
}


// MARK: - 数据库查询
extension SRStep {

    static func queryToday() -> SRStep {
        let total = SRStep()
        let dateStr = total.date.dateStr
        if let data = BandRecord.queryColumn(dateStr, .step) as? Data {
            total.parse(data)
        }
        return total
    }
    
    public static func query(with date: Date) -> SRStep? {
        let dateStr = BandRecord.dateStr(date)
        if let data = BandRecord.queryColumn(dateStr, .step) as? Data {
            let total = SRStep()
            total.parse(data)
            return total
        }
        return nil
    }
    
    public static func query(from fromDate: Date, to toDate: Date) -> [SRStep] {
        var items = [SRStep]()
        let fromDateStr = BandRecord.dateStr(fromDate)
        let toDateStr = BandRecord.dateStr(toDate)
        let records = BandRecord.queryRecord(from: fromDateStr, to: toDateStr, column: .step) as? [Data]
        records?.forEach({ (data) in
            let item = SRStep()
            item.parse(data)
            items.append(item)
        })
        return items
    }
}

