//
//  SRTotal.swift
//  BandKit
//
//  Created by NEIL on 2020/5/12.
//

import Foundation

public class SRTotal: SRModelBase, SRModelProtocol {
    // 日期
    public private(set) var date = SRFlashDateItem()
    /// 步数
    public private(set) var steps: UInt32 = 0
    /// 卡路里
    public private(set) var calories: UInt16 = 0
    /// 距离 （10 m）
    public private(set) var distance: UInt16 = 0
    /// 心率
    public private(set) var latestHeart: UInt8 = 0
    /// 运动时间（分钟）
    public private(set) var exerciseMinutes: UInt16 = 0
    /// 睡眠时间（分钟）
    public private(set) var sleepMinutes: UInt16 = 0
    
    ///: - Initialize
    required public init() {}

    ///: - override
    
    override var dataLen: Int { 16 }

    override func data() -> Data {
        let buffer = SRByteBuffer()
        buffer.putData(date.data())
        buffer.putShort(UInt16(steps & 0xFFFF))
        buffer.putShort(calories)
        buffer.putShort(distance)
        buffer.putShort(exerciseMinutes)
        buffer.putShort(sleepMinutes)
        buffer.putByte(UInt8((steps>>16) & 0xFF))
        buffer.putByte(latestHeart)
        return buffer.bufferData
    }

    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false}
        let buffer = SRByteBuffer(with: data)

        date.parse(buffer.getBytes(4))
        steps = UInt32(buffer.getShort())
        calories = buffer.getShort()
        distance = buffer.getShort()
        exerciseMinutes = buffer.getShort()
        sleepMinutes = buffer.getShort()
        steps += UInt32(buffer.get())<<16
        latestHeart = buffer.get()

        return true
    }
  
}

extension SRTotal: CustomStringConvertible {
    public var description: String {
        """
        {"date": \(date), "steps": \(steps), "calories": \(calories), "distance": \(distance),
         "sleepMinutes": \(sleepMinutes), "latestHeart": \(latestHeart)}
        """
    }
}


// MARK: - CSV 文件记录文本
extension SRTotal {
    public static func csvTitle() -> String {
        let titleItems: [String] = ["Date", "Steps", "Calories", "Distance", "Sleep minutes", "Exercise minutes", "Latest heart rate"]
        return titleItems.joined(separator: ",")
    }
    
    public func csvText() -> String {
        let text: [String] = ["\(date)", "\(steps)", "\(calories)", "\(distance)", "\(latestHeart)", "\(sleepMinutes)", "\(exerciseMinutes)"]
        return text.joined(separator: ",")
    }
}

extension SRTotal {
    public static var debug: SRTotal {
        let model = SRTotal()
        model.steps =  UInt32(random(min: 3600, max: 18000))
        model.calories = UInt16(0.0437 * Float(model.steps))
        model.distance = UInt16(0.0645 * Float(model.steps))
        model.sleepMinutes = 0
        model.exerciseMinutes = 0
        model.latestHeart = 0
        return model
    }
}

// MARK: - 数据库查询
extension SRTotal {
    
    public static func queryToday() -> SRTotal {
        let total = SRTotal()
        let dateStr = total.date.dateStr
        if let data = BandRecord.queryColumn(dateStr, .total) as? Data {
            total.parse(data)
        }
        return total
    }
    
    public static func query(with date: Date) -> SRTotal? {
        let dateStr = BandRecord.dateStr(date)
        if let data = BandRecord.queryColumn(dateStr, .total) as? Data {
            let total = SRTotal()
            total.parse(data)
            return total
        }
        return nil
    }

    public static func query(from fromDate: Date, to toDate: Date) -> [SRTotal] {
        var items = [SRTotal]()
        let fromDateStr = BandRecord.dateStr(fromDate)
        let toDateStr = BandRecord.dateStr(toDate)
        let records = BandRecord.queryRecord(from: fromDateStr, to: toDateStr, column: .total) as? [Data]
        records?.forEach({ (data) in
            let item = SRTotal()
            item.parse(data)
            items.append(item)
        })
        return items
    }
}

