//
//  SRByteBuffer.swift
//  BandKit
//
//  Created by Mac on 2020/5/4.
//

import Foundation

enum ByteOrder: Int {
    case bigEndian
    case litterEndian
}

typealias Byte = UInt8

typealias ByteArray = [UInt8]

class SRByteBuffer {
    
    // MARK: - Field
    let byteOrder: ByteOrder
    
    var bufferData: Data

    var position: Int
    
    var remain: Int { return limit - position }
    
    private var limit: Int { return bufferData.count }
    
    init(with data: Data=Data(), order: ByteOrder = .litterEndian) {
        self.bufferData = data
        self.byteOrder = order
        self.position = 0
    }
    
    convenience init(with byteArray: ByteArray, order: ByteOrder = .litterEndian) {
        let data = Data(bytes: byteArray, count: byteArray.count)
        self.init(with: data, order: order)
    }
    
    ///: - put. method

    func putData(_ data: Data){
        self.bufferData.append(data)
    }
    
    func putBytes(_ bytes: ByteArray){
        let data = Data(bytes)
        self.bufferData.append(data)
    }

    func putByte(_ byte: UInt8){
        let data = Data([byte])
        bufferData.append(data)
    }
    
    func putShort(_ short: UInt16) {
        var temp = short
        if byteOrder == .bigEndian { temp = CFSwapInt16BigToHost(short) }
        let bytes = [UInt8(truncatingIfNeeded: temp),
                     UInt8(truncatingIfNeeded: temp>>8)];
        let data = Data(bytes)
        bufferData.append(data)
    }
    
    func putInt32(_ value: UInt32) {
        var temp = value
        if byteOrder == .bigEndian { temp = CFSwapInt32HostToBig(value) }
        let bytes = [UInt8(truncatingIfNeeded: temp),
                     UInt8(truncatingIfNeeded: temp>>8),
                     UInt8(truncatingIfNeeded: temp>>16),
                     UInt8(truncatingIfNeeded: temp>>24)];
        let data = Data(bytes)
        bufferData.append(data)
    }

    ///: - get. method
    func getData(_ length: Int) -> Data {
        if length <= 0 { return Data() }
        if position+length > limit { BandLog.e("overflow"); return Data() }
        let endPosition = position + length;
        let bytes = bufferData.subdata(in: position..<endPosition)
        position += length
        return bytes
    }

    func get() -> UInt8 {
        if position > limit { BandLog.e("overflow"); return 0 }
        let bytes = [UInt8](bufferData)
        let result = bytes[position]
        position += 1
        return result
    }
    
    func getShort() -> UInt16 {
        if position+2 > limit { BandLog.e("overflow"); return 0 }
        let bytes = [UInt8](bufferData)
        let value = UInt16(truncatingIfNeeded: bytes[position+0]) +
                    UInt16(truncatingIfNeeded: bytes[position+1])<<8
        position += 2
        var result = value
        if byteOrder == .bigEndian { result = CFSwapInt16BigToHost(value) }
        return result
    }
    
    func getInt32() -> UInt32 {
        if position+4 > limit { BandLog.e("overflow"); return 0 }
        let bytes = [UInt8](bufferData)
        let value = UInt32(truncatingIfNeeded: bytes[position+0]) +
                    UInt32(truncatingIfNeeded: bytes[position+1])<<8  +
                    UInt32(truncatingIfNeeded: bytes[position+2])<<16 +
                    UInt32(truncatingIfNeeded: bytes[position+3])<<24
        position += 4
        var result = value
        if byteOrder == .bigEndian { result = CFSwapInt32HostToBig(value) }
        return result
    }
    
    func getBytes(_ length: Int) -> Data {
        if length==0 { return Data() }
        if position+length > limit { BandLog.e("overflow"); return Data() }
        let endPosition = position + length;
        let bytes = bufferData[position..<endPosition]
        position += length
        return bytes

    }
}
