//
//  SRFormat.swift
//  BandKit
//
//  Created by Mac on 2020/5/8.
//

import Foundation

public enum SRHourlyType: Int, SREnumtProtocol{
    case h12
    case h24

    var value: UInt8 {
        switch self {
        case .h12: return 0x01
        case .h24: return 0x00
        }
    }

    static func parse(_ byte: UInt8)-> SRHourlyType {
        let value = byte & 0x0F
        var type: SRHourlyType = .h24
        if value == SRHourlyType.h12.value { type = .h12 }
        return type
    }
}

public enum SRSystemType: Int, SREnumtProtocol {
    case metric
    case imperial
    
    var value: UInt8 {
        switch self {
        case .metric: return 0x00
        case .imperial: return 0x02
        }
    }
    

    static func parse(_ byte: UInt8)-> SRSystemType {
        let value = byte & 0x0F
        var type: SRSystemType = .metric
        if value == SRSystemType.imperial.value { type = .imperial }
        return type
    }
}

extension SRSystemType: CustomStringConvertible {
    public var description: String {
        switch self {
        case .metric: return "metric"
        case .imperial: return "imperial"
        }
    }
}

public enum SRTempUnitsType: Int, SREnumtProtocol {
    case centigrade
    case fahrenheit
    
    var value: UInt8 {
        switch self {
        case .centigrade: return 0x00
        case .fahrenheit: return 0x04
        }
    }
    
    static var collection:[SRTempUnitsType]  = [.centigrade, .fahrenheit]

    static func parse(_ byte: UInt8)-> SRTempUnitsType {
        
        let value = byte & 0x0F
        var type: SRTempUnitsType = .centigrade
        if value == SRTempUnitsType.fahrenheit.value { type = .fahrenheit }
        return type
    }
}

extension SRTempUnitsType: CustomStringConvertible {
    public var description: String {
        switch self {
        case .centigrade: return "Centigrade"
        case .fahrenheit: return "Fahrenheit"
        }
    }
}


public enum SRStaticCalorieSwitch: Int, SREnumtProtocol {
    case on
    case off
    
    var value: UInt8 {
        switch self {
        case .on: return 0x08
        case .off: return 0x00
        }
    }
    

    static func parse(_ byte: UInt8)-> SRStaticCalorieSwitch {
        let value = byte & 0x0F
        var type: SRStaticCalorieSwitch = .off
        if value == SRStaticCalorieSwitch.on.value { type = .on }
        return type
    }
}

extension SRStaticCalorieSwitch: CustomStringConvertible {
    public var description: String {
        switch self {
        case .on: return "On"
        case .off: return "Off"
        }
    }
}


/// 设备中对应的语言代码值
public enum SRLanType: Int, SREnumtProtocol {
    case en
    case zh
    case ja
    case de
    case it
    case ko
    case ro
    case fr
    case es
    case pt
    case ru
    case th
    case hu
    case tr
    case sv
    
    var value: UInt8 {
        switch self {
        case .en: return 0x00
        case .zh: return 0x10
        case .ja: return 0x20
        case .de: return 0x30
        case .it: return 0x40
        case .ko: return 0x50
        case .ro: return 0x60
        case .fr: return 0x70
        case .es: return 0x80
        case .pt: return 0x90
        case .ru: return 0xA0
        case .th: return 0xB0
        case .hu: return 0xC0
        case .tr: return 0xD0
        case .sv: return 0xE0
        }
    }
    
    var prefixIdentify: String {
        switch self {
        case .en: return "en"
        case .zh: return "zh"
        case .ja: return "ja"
        case .de: return "de"
        case .it: return "it"
        case .ko: return "ko"
        case .ro: return "ro"
        case .fr: return "fr"
        case .es: return "es"
        case .pt: return "pt"
        case .ru: return "ru"
        case .th: return "th"
        case .hu: return "th"
        case .tr: return "tr"
        case .sv: return "sv"
        }
    }
    
    static var collection: [SRLanType]  = [.en, .zh, .ja, .de, .it, .ko, .ro, .fr,  .es, .pt, .ru, .th, .hu, .tr, .sv]
    
    static func parse(_ byte: UInt8)-> SRLanType {
        
        let value = byte & 0xF0
        var type: SRLanType = .en
        for e in collection { if e.value==value { type = e; break } }
        return type
    }
    
    public static var app: SRLanType {
        guard let identify = Locale.preferredLanguages.first else { return.en }
        var type: SRLanType = .en
        for e in collection { if identify.hasPrefix(e.prefixIdentify) { type = e; break } }
        return type
    }
}

extension SRLanType: CustomStringConvertible {
    public var description: String {
        switch self {
        case .en: return "英文"
        case .zh: return "中文"
        case .ja: return "日文"
        case .de: return "德文"
        case .it: return "意大利文"
        case .ko: return "韩文"
        case .ro: return "罗马西亚文"
        case .fr: return "法文"
        case .es: return "西班牙文"
        case .pt: return "葡萄牙文"
        case .ru: return "俄文"
        case .th: return "语"
        case .hu: return "匈牙利文"
        case .tr: return "土耳其文"
        case .sv: return "瑞典文"
        }
    }
}

public class SRFormat: SRModelBase, SRModelProtocol {
    /// 小时制 , 默认（h24）
    public var hourly: SRHourlyType = .h24
    /// 公英制 — 默认（metric）
    public var system: SRSystemType = .metric
    /// 温度单位 — 默认（centigrade）
    public var tempUnit: SRTempUnitsType = .centigrade
    /// 静态卡路里计算 — 默认（off）
    public var staticcalorie: SRStaticCalorieSwitch = .off
    /// 语言类型
    public var lanType: SRLanType = .en
    
    ///: - Initialize
    required public init() {}
    
    ///: - internal
    var value: UInt8 {
        get { (hourly.value | system.value | tempUnit.value | staticcalorie.value | lanType.value) }
        set {
            self.hourly = SRHourlyType.parse(newValue)
            self.system = SRSystemType.parse(newValue)
            self.tempUnit = SRTempUnitsType.parse(newValue)
            self.staticcalorie = SRStaticCalorieSwitch.parse(newValue)
            self.lanType = SRLanType.parse(newValue)
        }
    }

    
    ///: - override

    override var dataLen: Int { 1 }
    
    override func data() -> Data {
        return Data([value])
    }
    
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false }
        let buffer = SRByteBuffer(with: data)
        value = buffer.get()
        return true
    }
    
}
