//
//  SRUserInfo.swift
//  BandKit
//
//  Created by Mac on 2020/5/8.
//

import Foundation

public enum SRBrightScreenTimeType: Int, SREnumtProtocol{
    case short, medium, long
    var value: UInt8 {
        switch self {
        case .short: return 0
        case .medium: return 1
        case .long: return 2
        }
    }
    
    static func parse(_ byte: UInt8)-> SRBrightScreenTimeType {
        var type: SRBrightScreenTimeType = .long
        if byte == short.value { type = .short }
        if byte == medium.value { type = .medium }
        if byte == long.value { type = .long }
        return type
    }
}


public class SRUserInfo: SRModelBase, SRModelProtocol {
    /// 单位、语言
    public var format: SRFormat = SRFormat()
    /// 用户
    public var user: SRUser = SRUser()
    /// 达标值
    public var goal: SRGoal = SRGoal()
    /// 达标铃声
    public var reachRing: SRReachRing = SRReachRing()
    /// 运动心率报警
    public var heartRateAlarm: SRAlarmExercise = SRAlarmExercise()
    /// 消息推送黑名单
    public var ancsBlacks: SRBlackList = SRBlackList()
    
    ///: - Feature
    /// 静态心率 检测
    public var restingHeartRate: SRFeature = SRFeature(feature: 0, startHour: 0, endHour: 23)
    /// 翻腕亮屏
    public var rollBrightScrren = SRFeature(feature: 0, startHour: 0, endHour: 23)
    
    ///: - Remind
    /// 来电提醒
    public var callRemind: SRRemind = SRRemind(feature: 1, startHour: 0, endHour: 23)
    /// 消息提醒
    public var messageRemind: SRRemind = SRRemind(feature: 1, startHour: 0, endHour: 23)
    /// 久坐提醒
    public var sedentaryRemind: SRRemind = SRRemind(feature: 0, startHour: 9, endHour: 18)
    
    
    ///: - reserve 保留字节
    /// 长度为 1
    /// 亮屏时间
    public var brightScreenTime: SRBrightScreenTimeType = .short
    
    /// - Abyx  长度为 9，
    public var isAbys: Bool = false
    /// 喝水提醒 - Abyx 使用， 无 Vibrate
    public var waterRemind: SRRemindInterval = SRRemindInterval()
    
        
    ///: - Initialize
    required public init() {}
    
    ///: - internal
    /// 表盘样式  默认：0
    var dialStyle: UInt8 = 0
    /// 电池电量（1～4）
    var batteryLevel: UInt8 = 0
    /// 保留字节 Abyx: 长度为 9， Default 长度为 1
    var reserve: [UInt8] = [0]
    /// Abyx: 长度为 9，

    var dataLen_abyx: Int { 48 }

    ///: - override
    
    override var dataLen: Int { 40 }
    
    override func data() -> Data {
        updateReserve()
        
        let buffer = SRByteBuffer()
        buffer.putByte(self.format.value)
        buffer.putData(self.user.data())
        buffer.putData(self.goal.data())
        buffer.putData(self.reachRing.data())
        
        buffer.putData(self.callRemind.data())
        buffer.putData(self.messageRemind.data())
        buffer.putData(self.restingHeartRate.data())
        buffer.putData(self.heartRateAlarm.data())
        buffer.putData(self.sedentaryRemind.data())
        buffer.putData(self.rollBrightScrren.data())
        
        buffer.putByte(self.dialStyle)
        buffer.putByte(self.batteryLevel)
        buffer.putByte(self.user.age)
        
        buffer.putData(self.ancsBlacks.data())
        buffer.putBytes(self.reserve)
        
        let data = buffer.bufferData
        return data
    }
    
    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen || data.count == dataLen_abyx else { return false}
        if data.count == dataLen { self.isAbys = false } else
        if data.count == dataLen_abyx { self.isAbys = true }
        
        let buffer = SRByteBuffer(with: data)
        self.format.value = buffer.get()
        self.user.parse(buffer.getBytes(3))
        self.goal.parse(buffer.getBytes(5))
        self.reachRing.parse(buffer.getBytes(3))
        
        self.callRemind.parse(buffer.getBytes(4))
        self.messageRemind.parse(buffer.getBytes(4))
        self.restingHeartRate.parse(buffer.getBytes(3))
        self.heartRateAlarm.parse(buffer.getBytes(4))
        self.sedentaryRemind.parse(buffer.getBytes(4))
        self.rollBrightScrren.parse(buffer.getBytes(3))
        
        self.dialStyle = buffer.get()
        self.batteryLevel = buffer.get()
        self.user.age = buffer.get()
        
        self.ancsBlacks.parse(buffer.getBytes(2))
        self.reserve = [UInt8](buffer.getBytes(buffer.remain))
        parseReserve()
        SRUserInfo.readed = true
        
        return true
    }
    
    /// 保留字节处理
    private func parseReserve() {
        if isAbys, reserve.count==9 {
            self.waterRemind.feature = reserve[0]
            self.waterRemind.startHour = reserve[1]
            self.waterRemind.endHour = reserve[2]
            self.waterRemind.interval = reserve[3]
        } else if !isAbys, reserve.count == 1 {
            self.brightScreenTime = SRBrightScreenTimeType.parse(reserve[0])
        }
    }
 
    private func updateReserve() {
        if isAbys, reserve.count==9 {
            reserve[0] = self.waterRemind.feature
            reserve[1] = self.waterRemind.startHour
            reserve[2] = self.waterRemind.endHour
            reserve[3] = self.waterRemind.interval
        } else if !isAbys, reserve.count > 0 {
            reserve[0] =  self.brightScreenTime.value
        }
    }
}

// MARK: - CSV

extension SRUserInfo {
    func csvContent(_ userName: String) -> String {
        let brightScreenTimeContent = isAbys ? "" : ("Bright screen time" + ",\(brightScreenTime.value)\n")
        var waterContent = isAbys ? waterRemind.csvContent(name: "Water reminder", noVibrate: true) : ""
        var sedentaryIntervalContent = ""
        let drinkInfo = (SRDrinkInfo.readed && !isAbys) ? SRDataManager.drinkInfo : nil
        if let drink = drinkInfo {
            waterContent = drink.waterRemind.csvContent(name: "Water reminder")
            sedentaryIntervalContent = ",Intervl minutes" + ",\(drink.sedentaryInterval)\n"
        }
        
        var content = "Formart" + ",\(format.value)\n"
        content += brightScreenTimeContent
        content += "Dial style" + ",\(dialStyle)\n"
        content += "Battery level" + ",\(batteryLevel)\n"
        content += ancsBlacks.csvContent()
        content += user.csvContent(name: userName, drink: drinkInfo)
        content += goal.csvContent()
        content += reachRing.csvContent()
        content += heartRateAlarm.csvContent()
        content += callRemind.csvContent(name: "Incoming reminder")
        content += messageRemind.csvContent(name: "Message reminder")
        content += sedentaryRemind.csvContent(name: "Sedentary reminder")
        content += sedentaryIntervalContent
        content += restingHeartRate.csvContent(name: "Resting Heart rate detection")
        content += rollBrightScrren.csvContent(name: "Roll Bright scrren")
        content += waterContent
        return content
    }
    
    public static func csvContent(_ userName: String = "") -> String {
        return SRDataManager.userInfo.csvContent(userName)
    }
}


// MARK: - 数据缓存
extension SRUserInfo {
    func saveToCache() {
        SRUserInfo.default = self
    }
    
    static func readForCache() -> SRUserInfo {
        if let cache = SRUserInfo.default { return cache }
        return SRUserInfo()
    }
    
    private static let key = "SRUserInfoBandKitKey"
    private static var _default: SRUserInfo?
    private static var `default`: SRUserInfo? {
        get {
            if let info = _default { return info}
            let info = SRUserInfo()
            if let data = UserDefaults.standard.data(forKey: key) { info.parse(data) }
            _default = info
            return info
        }
        set {
            _default = newValue
            if let info = _default {
                UserDefaults.standard.set(info.data(), forKey: key)
            } else {
                UserDefaults.standard.removeObject(forKey: key)
            }
            UserDefaults.standard.synchronize()
        }
    }
}

/// 新连接设备是否读取过数据
extension SRUserInfo {
    private static let readedKey = "SRUserInfoReadedKey"
    public static var readed: Bool {
        get {
            UserDefaults.standard.bool(forKey: readedKey)
        }
        set {
            UserDefaults.standard.set(newValue, forKey: readedKey)
            UserDefaults.standard.synchronize()
        }
    }
}

extension SRUserInfo {
    
    /// : - 美国区域
    /// 用于默认华摄度设置
    static var country_US: Bool {
        let locale = NSLocale(localeIdentifier: Locale.current.identifier)
        guard let code:String = locale.object(forKey: .countryCode) as? String else { return false }
        
        let items = ["US", "UM", "VI", "AS"]
        if items.contains(code) {
            return true
        }
        return false
    }

}
