//
//  SRWeatherForecast.swift
//  BandKit
//
//  Created by Mac on 2020/7/7.
//

import Foundation
import NEUtils

// MARK: - 天气预报 - 24 小时

public class SRWeatherForecastHour: SRModelProtocol {
    /// 第一组对应的起始时间戳
    /// 时间戳，是相对于1970-1-1的时间 单位 秒
    public var time: TimeInterval = 0
    
    /// 24 组小时项
    public var hourItems: [NEWeatherHourItem] = []

    // MARK: - Initialize
    required public init() { }
    
    // MARK: - internal
    var dateTime: String { return dformatter.string(from: Date(timeIntervalSince1970: time)) }

    lazy var dformatter: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "yyyy-MM-dd HH"
           return dformatter
    }()
    
    var vaild: Bool {
        let hourStartDate = Date(timeIntervalSince1970: time)
        let nowTimeinterval = Date().timeIntervalSince1970 + 1800
        let nowDate = Date(timeIntervalSince1970: nowTimeinterval)
        let components = SRCalendarUtils.components([.hour], from: SRCalendarUtils.hourDate(for: hourStartDate), to: SRCalendarUtils.hourDate(for: nowDate))
        guard abs(components.hour!) < 24, hourItems.count >= 24 else { return false}
        return true
    }
}

import HandyJSON
extension SRWeatherForecastHour {
    
    public func mapping(mapper: HelpingMapper) {
        mapper <<<
        self.time <-- TransformOf<TimeInterval, String>(fromJSON: { [unowned self] (rawString) -> (TimeInterval) in
            let date = self.dformatter.date(from: rawString ?? "") ?? Date()
            return date.timeIntervalSince1970
        }, toJSON: { [unowned self] (_) -> String? in
            return self.dateTime
        })
    }
}

// MARK: -  Data format for protocol

extension SRWeatherForecastHour {
    
    /// 转换为协议格式数据
    func data() -> Data? {
        // guard vaild else { return nil }

        let date = Date(timeIntervalSince1970: time)
        let components =  SRCalendarUtils.components(ofDate: date)
        let year    = UInt16(components.year!)
        let month   = UInt8(components.month!)
        let day     = UInt8(components.day!)
        let hour    = UInt8(components.hour!)
        
        let buffer = SRByteBuffer()
        buffer.putShort(year)
        buffer.putByte(month)
        buffer.putByte(day)
        buffer.putByte(hour)
        
        /// 保留字节
        let tbd: [UInt8] = [0, 0, 0]
        buffer.putBytes(tbd)

        for idx in 0..<24 {
            let item = hourItems[idx]
            buffer.putByte(UInt8(truncatingIfNeeded: item.type.rawValue))
            buffer.putByte(UInt8(truncatingIfNeeded: item.temperature))
            buffer.putByte(UInt8(truncatingIfNeeded: item.humidity))
            buffer.putByte(UInt8(truncatingIfNeeded: item.sunuv))
            buffer.putByte(UInt8(truncatingIfNeeded: item.windSpeed))
            buffer.putByte(UInt8(truncatingIfNeeded: item.windDirection))
        }
        return buffer.bufferData
    }
}


// MARK: - 天气预报 - 15 天

public class SRWeatherForecastDay: SRModelProtocol {
    
    /// 城市名称 18 Byte
    public var cityName: String = ""
    /// 当前小时项
    public var currentHourItem: NEWeatherHourItem? = nil
    /// 当天项
    public var currentDayItem: NEWeatherDayItem? = nil
    /// 15 组 每天项 （明天作为起始）
    public var dayItems: [NEWeatherDayItem] = []

    // MARK: - Initialize
    required public init() { }

    
    // MARK: - internal
    var dateTime: String { return dformatter.string(from: Date()) }

    lazy var dformatter: DateFormatter = {
           let dformatter = DateFormatter()
           dformatter.dateFormat = "yyyy-MM-dd"
           return dformatter
    }()
    
    var vaild: Bool {
        guard dayItems.count >= 15, currentHourItem != nil, currentDayItem != nil  else { return false}
        return true
    }
}

// MARK: -  Data format for protocol

extension SRWeatherForecastDay {
    
    /// 转换为协议格式数据
    func data() -> Data? {
        guard vaild else { return nil }
        
        let date = Date()
        let components =  SRCalendarUtils.components(ofDate: date)
        let year    = UInt16(components.year!)
        let month   = UInt8(components.month!)
        let day     = UInt8(components.day!)
        let weekDay = UInt8(components.weekday!)
        
        let buffer = SRByteBuffer()
        buffer.putShort(year)
        buffer.putByte(month)
        buffer.putByte(day)
        buffer.putByte(weekDay)
        
        var nameLen = 0
        var nameBytes: [UInt8] = [0,0,0,0,0,0,  0,0,0,0,0,0,  0,0,0,0,0,0]
        if let nameData = cityName.data(using: .utf8) {
            let bytes = [UInt8](nameData).prefix(nameBytes.count)
            nameLen = bytes.count
            nameBytes.replaceSubrange(0..<nameLen, with: bytes)
        }
        buffer.putByte(UInt8(nameLen))
        buffer.putBytes(nameBytes) // 24 Byte
        
        buffer.putByte(UInt8(truncatingIfNeeded: currentHourItem!.type.rawValue))
        buffer.putByte(UInt8(truncatingIfNeeded: currentHourItem!.temperature))
        
        buffer.putByte(UInt8(truncatingIfNeeded: currentDayItem!.temperatureMax))
        buffer.putByte(UInt8(truncatingIfNeeded: currentDayItem!.temperatureMin))
        
        buffer.putByte(UInt8(truncatingIfNeeded: currentHourItem!.humidity))
        buffer.putByte(UInt8(truncatingIfNeeded: currentHourItem!.windDirection))
        buffer.putByte(UInt8(truncatingIfNeeded: currentHourItem!.windSpeed))
        buffer.putByte(UInt8(truncatingIfNeeded: currentHourItem!.sunuv))
        buffer.putShort(UInt16(truncatingIfNeeded: currentHourItem!.airPressure))
        
        buffer.putByte(UInt8(truncatingIfNeeded: currentDayItem!.sunrise/60))
        buffer.putByte(UInt8(truncatingIfNeeded: currentDayItem!.sunrise%60))
        buffer.putByte(UInt8(truncatingIfNeeded: currentDayItem!.sunset/60))
        buffer.putByte(UInt8(truncatingIfNeeded: currentDayItem!.sunset%60)) // 38 Byte
        
        /// 保留字节
        let tbd: [UInt8] = [0, 0]
        buffer.putBytes(tbd)

        for idx in 0..<15 {
            let item = dayItems[idx]
            buffer.putByte(UInt8(truncatingIfNeeded: item.temperatureMax))
            buffer.putByte(UInt8(truncatingIfNeeded: item.temperatureMin))
            buffer.putByte(UInt8(item.type.rawValue))
            buffer.putByte(UInt8(item.humidity))
        }
        return buffer.bufferData
    }
}


extension SRWeatherForecastHour {
    
    /// 未来24小时小时天气 预测
    public static var forecastHour: SRWeatherForecastHour? {
        guard let items = NEWeatherInfo.hourItems else { return nil}
        let forecast = SRWeatherForecastHour()
        forecast.time = NEWeatherInfo.hourly_ts
        /// 服务器新获取的数据，hourly_ts 可能大于当前时间一点点，即 offsetHour = -1
        /// 为了避免这种情况天气数据不能实时显示：
        /// hourly_ts 需减去一个小时，hourly数组左复制一组
        if NEWeatherInfo.currentHourOffset == -1 { forecast.time -= 3600 }
        forecast.hourItems = items
        // BandLog.v(forecast.toJSONString()!)
        return forecast
    }
}

extension SRWeatherForecastDay {

    /// 未来15天天气  预报
    public static var forecastDay: SRWeatherForecastDay? {
        guard let hasCurrentHourItem = NEWeatherInfo.currentHourItem,
              let hasCurrentDayItem = NEWeatherInfo.currentDayItem else { return nil}

        let offsetDay = NEWeatherInfo.currentDayOffset
        guard offsetDay >= 0 else { return nil }
        
        guard let items: [NEWeatherDayItem] = NEWeatherInfo.dayItems else { return nil }
        let forecastItems: [NEWeatherDayItem] = items.suffix(items.count-1-offsetDay)
        guard forecastItems.count >= 15 else { return nil }
        
        let forecast = SRWeatherForecastDay()
        forecast.currentHourItem = hasCurrentHourItem
        forecast.currentDayItem = hasCurrentDayItem
        forecast.dayItems = forecastItems
        forecast.cityName = NEWeatherTask.cityName
        // BandLog.v(forecast.toJSONString()!)
        return forecast
    }
}
