//
//  CmdHolder.swift
//  BandKit
//
//  Created by Mac on 2020/4/29.
//

import Foundation
import Repeater

typealias ReponseClosure = (CmdReponse)->Void
typealias TimeoutClosure = ()->Void

/// 命令模型结构

class CmdHolder {
    
    init(type: SRCommandType, data: Data=CmdHolder.defaultData, time: TimeInterval = 8, reponse: ReponseClosure? = nil) {
        self.type = type
        self.describe = type.name
        self.data = data
        self.time = time
        self.reponse = reponse
        self.packetsParse()
    }
    
    // MARK: - Field
    let type: SRCommandType
    var packets: [Data] = []
    var describe: String
    var parseText: String? = nil
    
    private let data: Data
    private let time: TimeInterval
    private let reponse: ReponseClosure?

    ///: -  Error
    private var bleoffReponse: CmdReponse { CmdReponse(type, error: SRRequestError.bleoff) }
    private var diconnectedReponse: CmdReponse { CmdReponse(type, error: SRRequestError.disconnected) }
    private var sendFailedReponse: CmdReponse { CmdReponse(type, error: SRRequestError.sendfailed) }
    private var timeoutReponse: CmdReponse { CmdReponse(type, error: SRRequestError.timeout) }
    private var cancelReponse: CmdReponse { CmdReponse(type, error: SRRequestError.cancel) }
    private var updateReponse: CmdReponse { CmdReponse(type, error: SRRequestError.update) }
    private var noneReponse: CmdReponse { CmdReponse(type, error: SRRequestError.noreponse) }
    
    ///: - 超时
    private var timeoutClosure: TimeoutClosure?
    private lazy var timeoutRepeater: Repeater = {
        let repeater = Repeater.once(after: .seconds(time)) {  (_) in
            cmdSyncAction { [weak self] in
                guard let strong = self else { return }
                strong.onTimeout()
                strong.timeoutClosure?()
            }
        }
        repeater.pause()
        return repeater;
    } ()
    

    // MARK: - Method for internal
    /// 发送完成， 所有的数据包发送成功后调用
    /// 1. 打印发送的数据
    /// 2. time > 0 时:  在此设置超时定时器，返回 true
    func onSendCompleted(_ closure: @escaping TimeoutClosure) -> Bool {
        BandLog.v(sendSuccessLog)
        timeoutClosure = closure
        guard time > 0 else { onNoneReponse(); return false }
        timeoutRepeater.start()
        return true
    }
    
    /// 命令响应完成
    func onReponseCompleted(_ recv: CmdReponse) {
        guard time > 0.1 else { return}
        timeoutRepeater.pause()
        reponse?(recv)
    }
    
    /// 蓝牙关闭
    func onBleoff() {
        reponse?(bleoffReponse)
    }
    
    /// 断开连接
    func onDisconnected() {
        reponse?(diconnectedReponse)
    }
    
    /// 发送失败
    func onSendFailed() {
        BandLog.e(sendFailedLog)
        reponse?(sendFailedReponse)
    }
    /// 响应超时
    func onTimeout() {
        reponse?(timeoutReponse)
    }
    /// 取消命令
    func onCancel() {
        reponse?(cancelReponse)
    }
    /// 更新命令
    func onUpdate() {
        reponse?(updateReponse)
    }
    /// 命令完成 - 无响应
    func onNoneReponse() {
        reponse?(noneReponse)
    }
    
    
    // MARK: - Private methods    
    /// 将数据打包成 命令格式包
    private func packetsParse(){
        let maxPacketLen: UInt8 = UInt8(min((CmdHolder.maxLenResponseWithout - 4), 251))
        let length = self.data.count
        let count = Int( ceilf( Float(length) / Float(maxPacketLen) ))
        
        var packetNo: UInt8 = 0
        var packetLen: UInt8 = 0
        var remainderLen = length
        var position = 0
        
        self.packets.removeAll()
        for i in 0..<count {
            packetNo = UInt8(i + 1) & 0xff
            if i == count-1 { // 最后一个包
                //长度等于剩余的数据长度
                packetLen = UInt8(remainderLen) & 0xff
                // 将 packetNo 最高位设为1，表示数据结束
                packetNo |= 0x80;
            } else {
                packetLen = maxPacketLen;
            }
            
            let endPosition = position + Int(packetLen);
            let packetData = self.data[position..<endPosition]
            let packet = createPacket(with: packetData, packetNo)
            self.packets.append(packet)
            
            position += Int(packetLen);
            remainderLen -= Int(packetLen);
        }
    }
    
    private func createPacket(with packetData: Data, _ packetNo: UInt8) -> Data{
        let head: [UInt8] = [type.cmd, type.ext, packetNo, UInt8(packetData.count)]
        var packet = Data(head)
        packet.append(packetData)
        return packet
    }
}

extension CmdHolder {
    static var defaultData =  Data([0x00])
    static var maxLenResponse: UInt16 = 20
    static var maxLenResponseWithout: UInt16 = 20
}

extension CmdHolder: Equatable {
    static func == (lhs: CmdHolder, rhs: CmdHolder) -> Bool {
        return (lhs.type == rhs.type)
    }
    
    static func === (lhs: CmdHolder, rhs: CmdHolder) -> Bool {
        return (lhs.type == rhs.type && lhs.data == rhs.data)
    }
}

// MARK: - Logger 日记

extension CmdHolder {
    private var textSended: String { "Sended: \(describe)" } // 发送指令
    /// 发送的命令包
    private var textPacket: String {
        guard packets.count<3 else { return "Packets: [1...\(packets.count)]" }
        return "Packets: \(packets.logDescription)"
    }
    /// 发送数据解析
    private var textParsed: String? {
        guard let hasParsedText = parseText else { return nil }
        return "Parsed: \(hasParsedText)"
    }
    private var sendSuccessLog: String {
        var textLogger: String = "\(textSended)\n\(textPacket)"
        if let text = textParsed { textLogger += "\n\(text)"}
        return textLogger
    }
    private var sendFailedLog: String {
        let textLogger: String = "\(textSended)"
        return textLogger + "\nError: writeCharacteristic is failed"
    }
}



