//
//  CmdRequest.swift
//  BandKit
//
//  Created by Mac on 2020/4/29.
//

import Foundation
import RxSwift

class CmdRequest {
    typealias Queue = CmdQueue
    typealias ReponseClosure = (CmdReponse)->Void

    /// 活跃队列：发送等待回应的命令队列
    private var activeQueue = Queue()
    /// 保留队列：还未发送的命令队列
    private var reservedQueue = Queue()
    /// 是否有活跃命令
    private var isActive : Bool { return activeQueue.count>0 }
    
    /// - subjects
    let disposeBag = DisposeBag()
    let receivedSubject = PublishSubject<CmdReponse>()
    
    
    // MARK: - 队列操作
    
    /// 直接加入到队尾
    /// - Parameter cmd: 命令对象
    private func push(_ cmd:CmdHolder) {
        cmdSyncAction {
            if self.isActive {
                self.reservedQueue.push(cmd)
            } else {
                self.send(cmd)
            }
        }
    }
    
    
    /// 直接插入到队头
    /// - Parameter cmd: 命令对象
    private func insertHeader(_ cmd:CmdHolder) {
        cmdSyncAction {
            if self.isActive {
                self.reservedQueue.insertHeader(cmd)
            } else {
                self.send(cmd)
            }
        }
    }
    
    /// 更新加入到队尾
    /// - Parameter cmd: 命令对象
    private func updatePush(_ cmd:CmdHolder) {
        cmdSyncAction {
            if self.isActive {
                self.reservedQueue.updatePush(cmd)
            } else {
                self.send(cmd)
            }
        }
    }
    
    
    /// 直接插入到队头
    /// - Parameter cmd: 命令对象
    private func updateInsertHeader(_ cmd:CmdHolder) {
        cmdSyncAction {
            if self.isActive {
                self.reservedQueue.updateInsertHeader(cmd)
            } else {
                self.send(cmd)
            }
        }
    }
    
    
    /// 强制发送命令
    /// - Parameter cmd: 命令对象
    private func enforce(_ cmd:CmdHolder) {
        cmdSyncAction {
            self.send(cmd)
        }
    }
    
    private func clear() {
        cmdSyncAction {
            self.activeQueue.clear()
            self.reservedQueue.clear()
        }
    }

    
    // MARK: - 发送命令
    /// 先把命令放到活跃队列，然后发送命令数据包
    /// - Parameter cmd: 命令对象
    private func send(_ cmd: CmdHolder) {
        /// 用于标记发送数据包的结果
        var result = false
        
        /// 发送完整的包格式
        activeQueue.push(cmd)
        
        guard SRKitService.centralState == .poweredOn else { onSendFailed(cmd, bleoff: true);  return; }
        guard SRKitService.isConnected else { onSendFailed(cmd, disconnected: true);  return; }
        for data in cmd.packets {
            result = SRKitService.writeValue(with: data)
            if !result { break }
        }
        
        /// 发送失败
        guard result else { onSendFailed(cmd); return;}
        
        /// 发送成功 - 等待发送下一条
        /// 超时回调 - 发送下一条
        let hasTimeoutClosure = cmd.onSendCompleted { [weak self] in
            guard let strong = self else {return}
            strong.onSendNext(cmd)
        }
        /// 无响应 - 立即发送下一条指令
        if !hasTimeoutClosure { onSendNext(cmd) }
            
    }
    
    /// 发送系一条指令
    private func onSendNext(_ cmd: CmdHolder? = nil) {
        if let hasCmd = cmd { activeQueue.pop(hasCmd) }
        if !isActive, let item = reservedQueue.pop() {
            send(item)
        }
    }
    
    
    
    /// 发送失败
    private func onSendFailed(_ cmd: CmdHolder, bleoff: Bool = false, disconnected: Bool = false) {
        if bleoff {
            cmd.onBleoff()
        } else if disconnected {
            cmd.onDisconnected()
        } else {
            cmd.onSendFailed()
        }
        failedHandelr(cmd)
    }
    
    private func failedHandelr(_ cmd: CmdHolder) {
        activeQueue.pop(cmd)
        activeQueue.clear()
        reservedQueue.clear()
    }
    
    // MARK: - 接收数据
    
    /// 缓存接收数据，接收完后删除
    private var reponse = CmdReponse(0, 0, 0)
    
    private func received(_ data: Data) throws {
        
        /// 单包数据过滤
        guard data.count > 4 else { throw SRReponseDataError.tooshort }
        let bytes = [UInt8](data)
        
        let cmd   = bytes[0]
        let ext   = bytes[1]
        let pkNo  = bytes[2]
        let pkLen = bytes[3]
        let byte1 = bytes[4]
        
        let type = SRCommandType.parse(cmd, ext, byte1)
        let realNo = realPacketNo(pkNo)
        let realData = data[4..<data.count]

        /// 第一个包
        if realNo==1 { reponse = CmdReponse(cmd, ext, byte1) }
        /// 包 CommandType 不匹配
        guard type == reponse.type else {  throw SRReponseDataError.command }
        /// 包编号错误
        guard realNo == reponse.packets.count + 1 else {  throw SRReponseDataError.packetno }
        /// 包长度格式错误
        guard pkLen+4 == data.count else { throw SRReponseDataError.lenght }
        
        reponse.packets.append(data)
        if reponse.data == nil { reponse.data = Data()}
        reponse.data!.append(realData)
        
        /// 最后一个包
        if isLastPacket(pkNo) {
            checkError(reponse)
            receiveCompleted(reponse)
        }
    }
    
    /// 接收完整的数据后， 检测错误字段
    private func checkError(_ reponse: CmdReponse) {
        guard let data = reponse.data, data.count <= 2 else { return }
        if let type = reponse.type, case .devNotify(_) = type { return }

        let dataLen = data.count
        let bytes = [UInt8](data)
        let byte4 = bytes[0]
        reponse.code = SRReponseCode(rawValue: byte4) ?? .error

        /// 写数据应答
        if dataLen == 1, reponse.code == .error {
            reponse.error = SRReponseError.parse(byte4)
        }
        /// 读数据错误
        if dataLen == 2, reponse.code == .error {
            let byte5 = bytes[1]
            reponse.error = SRReponseError.parse(byte5)
        }
    }
    
    
    /// 接收完整的数据
    private func receiveCompleted(_ reponse: CmdReponse) {
        if reponse.error == nil { reponse.onParse() }
        if let type = reponse.type, let cmd = self.activeQueue.first(type) { reponse.describe = cmd.describe;  reponse.notify = false }
        reponse.print()
        /// 接收数据完成， 发送事件
        receivedSubject.onNext(reponse)
        /// 处理队列完成回调再执行下一条指令
        cmdSyncAction {
            if let type = reponse.type, let cmd = self.activeQueue.first(type) {
                reponse.describe = cmd.describe
                cmd.onReponseCompleted(reponse)
                self.onSendNext(cmd)
            }
        }
    }
}

// MARK: - 类方法、属性

extension CmdRequest {
    static let shared = CmdRequest();
    static let receivedOutput: PublishSubject<CmdReponse> = { shared.receivedSubject.asObserver() } ()
    static func received(_ data: Data) throws { try shared.received(data) }


    /// -- 命令
    static func push(_ cmd:CmdHolder) { shared.push(cmd) }
    static func insertHeader(_ cmd:CmdHolder) { shared.insertHeader(cmd) }
    static func updatePush(_ cmd:CmdHolder) { shared.updatePush(cmd) }
    static func updateInsertHeader(_ cmd:CmdHolder) { shared.updateInsertHeader(cmd) }
    static func enforce(_ cmd:CmdHolder) { shared.enforce(cmd) }
    static func clear() { shared.clear() }
}
