//
//  SRCmdUtils.swift
//  BandKit
//
//  Created by Mac on 2020/4/29.
//

import Foundation

// MARK: - ********** Public **********


public typealias SRWriteClosure = (Any?, Error?) ->Void

// MARK: - 建立连接转态
public enum SRConnectEstablishState {
    public enum FailedType {
        case bleoff, timeout, code(Int)
    }
    case successed, failed(FailedType)
}

// MARK: - 连接性状态
public enum SRConnectState {
    case connected, disconnected(String)
}


// MARK: - 数据发送请求失败

public enum SRRequestError: Error {
    case bleoff       // 蓝牙已关闭
    case disconnected // 连接断开
    case sendfailed   // 发送失败, 如：连接断开，服务中断异常
    case noreponse    // 不响应
    case timeout      // 响应超时

    case cancel       // 取消发送
    case update       // 更新请求
    case uploadInterrupt // 上传文件时，中断发送其他指令

    public var text: String {
        switch self {
        case .bleoff:       return "蓝牙已关闭"
        case .disconnected: return "连接断开"
        case .sendfailed:   return "发送失败"
        case .noreponse:    return "发送成功, 不响应"
        case .timeout:      return "响应超时"
        
        case .cancel:       return "取消发送"
        case .update:       return "更新请求"
        case .uploadInterrupt:    return "上传文件时，中断发送指令"
        }

    }
}

// MARK: - 响应数据出错
enum SRReponseDataError:Error{
    case tooshort   // 数据包太短
    case command    // 指令不匹配
    case packetno   // 包长度格式错误
    case lenght     // 命令验证
}

extension SRReponseDataError: CustomStringConvertible {
    public var description: String {
        switch self {
        case .tooshort: return "数据包太短"
        case .command:  return "指令不匹配"
        case .packetno: return "包序号错误"
        case .lenght:   return "数据长度错误"
        }
    }
}

// MARK: - 应答指令错误集

public enum SRReponseCode: UInt8 {
    /// 设备应答结果 bytes[4]
    case success = 0x00 // 指令发送并执行成功;
    case error   = 0x01 // 指令有错误
}


public enum SRReponseError: UInt8, Error {
    /// 如果 bytes[4] ==  error,  bytes[5] 是错误种类
    case nosupport  // Cmd+Ext_Cmd+ 长度都不对， 部分读指令附加字节不是00也会如此回复；
    case param      // 指令参数超出范围要求; 比如说写时间, 月份超过12这种类似错误;
    case length     // 指令数据长度不符合要求;
    case state      // 如设备断开、除去非应答指令, 其他出错统一回复这个;
    case delay      // 指令需要过一段时间执行完毕继续回复success, 先回复代表收到指令
    case notfound   // 一般代表没有相应的数据
    case unknown    // 未知错误
    var value: UInt8 {
        switch self {
        case .unknown:      return 0xFF
        case .nosupport:    return 0x02
        case .param:        return 0x03
        case .length:       return 0x04
        case .state:        return 0x05
        case .delay:        return 0x06
        case .notfound:     return 0x07
        }
    }
    
    static var collection: [SRReponseError] = [.unknown, .nosupport, .param, .length, .state, .delay, .notfound]

    static func parse(_ value: UInt8) -> SRReponseError {
        var error: SRReponseError = .unknown
        for e in collection { if e.value==value { error = e; break }}
        return error
    }
}

// MARK: - 连接类型

/// 设备的连接类型
public enum SRConnectType: CustomStringConvertible {
    case none   // 未连接
    case direct // 直连（未配对）
    case ios    // iOS 配对连接
    case android // Android 连接
    
    var value: UInt8 {
        switch self {
        case .none:     return 0
        case .direct:   return 1
        case .ios:      return 2
        case .android:  return 3
        }
    }
    
    var text: String {
        switch self {
        case .none:     return "none"
        case .direct:   return "direct"
        case .ios:      return "ios"
        case .android:  return "android"
        }
    }
    
    static var collection: [SRConnectType] = [.none, .direct, .ios, .android]

    static func parse(_ value: UInt8) -> SRConnectType {
        var type: SRConnectType = .none
        for e in collection { if e.value==value { type = e; break }}
        return type
    }
    
    public var description: String {
        return """
        {"pairWay": "\(text)"}
        """
    }
}



// MARK: - 传输速度类型

/// 蓝牙传输速度类型
public enum SRTransferSpeedType: CustomStringConvertible {
    case low    // 低速
    case high   // 高速
    
    var value: UInt8 {
        switch self {
        case .low:     return 0
        case .high:   return 1
        }
    }
    
    var text: String {
        switch self {
        case .low:  return "low"
        case .high: return "high"
        }
    }
    
    static var collection: [SRTransferSpeedType] = [.low, .high]

    static func parse(_ value: UInt8) -> SRTransferSpeedType {
        var type: SRTransferSpeedType = .low
        for e in collection { if e.value==value { type = e; break }}
        return type
    }
    
    public var description: String {
        return """
        {"transferSpeed": "\(text)"}
        """
    }
}


// MARK: - 设备事件通知类型
public enum SRNotifyType: CustomStringConvertible {
    ///: - 相机
    public enum Camera: CustomStringConvertible {
        case enter   // 进入
        case exit    // 退出
        case takePicture   // 拍照
        var ext: UInt8 {
            switch self {
            case .enter:        return 0x02
            case .exit:         return 0x03
            case .takePicture:  return 0x5A
            }
        }
        var byte1: UInt8 {
            switch self {
            case .enter:       return 0x00
            case .exit:        return 0x00
            case .takePicture: return 0x06
            }
        }
        public var description: String {
            switch self {
            case .enter:        return "进入"
            case .exit:         return "退出"
            case .takePicture:  return "拍照"
            }
        }
        
    }
    
    ///: - 查找手机
    public enum FindPhone: CustomStringConvertible {
        case start  // 开启
        case stop  // 停止

        var ext: UInt8 {
            return 0x5A
        }
        
        var byte1: UInt8 {
            switch self {
            case .start: return 0x0C
            case .stop:  return 0x0D
            }
        }
        
        public var description: String {
            switch self {
            case .start: return "开启"
            case .stop:  return "停止"
            }
        }
    }
    
    ///: - 训练
    public enum Exercise: CustomStringConvertible, Equatable {
        case start  // 开启
        case stop   // 停止
        case ignore // 忽略
        case sport(SRSportType)   // 运动类型
        var ext: UInt8 {
            switch self {
            case .start, .stop, .ignore: return 0x00
            case .sport(_):              return 0x02
            }
        }
        var byte1: UInt8 {
            switch self {
            case .start:    return 0x00
            case .stop:     return 0x01
            case .ignore:   return 0x02
            case .sport(let type): return type.value
            }
        }
        
        public var value: UInt16 {
            (UInt16(ext) << 8) + UInt16(byte1)
        }
       
        public static func parse(_ value: UInt16) ->  Exercise? {
            let ext:   UInt8 = UInt8(truncating: NSNumber(value: value>>8))
            let byte1: UInt8 = UInt8(truncating: NSNumber(value: value))
            switch ext {
            case 0x00:
                switch byte1 {
                case 0x00: return .start
                case 0x01: return .stop
                case 0x02: return .ignore
                default:  break
                }
            case 0x02:
                switch byte1 {
                case 0x00: break
                default: if let type = SRSportType.parse(byte1) { return .sport(type) };
                }
            default: break
            }
            return nil
        }
        
        public var description: String {
            switch self {
            case .start:  return "开启"
            case .stop:   return "停止"
            case .ignore: return "忽略"
            case .sport(let type): return "运动(\(type))"
            }
        }
        public static func == (lhs: Exercise, rhs: Exercise) -> Bool {
            lhs.ext == rhs.ext && lhs.byte1 == rhs.byte1
        }
    }
    
    case camera(Camera)
    case exercise(Exercise)
    case findPhone(FindPhone)
    var ext: UInt8 {
        switch self {
        case .camera(let type):    return type.ext
        case .exercise(let type):  return type.ext
        case .findPhone(let type): return type.ext
        }
    }
    var byte1: UInt8 {
        switch self {
        case .camera(let type):    return type.byte1
        case .exercise(let type):  return type.byte1
        case .findPhone(let type): return type.byte1
        }
    }
    static func parse(_ ext: UInt8, _ byte1: UInt8) -> SRNotifyType? {
        switch ext {
        case 0x00:
            switch byte1 {
            case 0x00: return .exercise(.start)
            case 0x01: return .exercise(.stop)
            case 0x02: return .exercise(.ignore)
            default:  break
            }
        case 0x02:
            switch byte1 {
            case 0x00: return .camera(.enter)
            default: if let type = SRSportType.parse(byte1) { return .exercise(.sport(type)) };
                
            }
        case 0x03: return .camera(.exit)
        case 0x04: return .findPhone(.start)
        case 0x5A:
            switch byte1 {
            case 0x06: return .camera(.takePicture)
            case 0x07: return .findPhone(.stop)
            case 0x0C: return .findPhone(.start)
            case 0x0D: return .findPhone(.stop)
            default:   break
            }
        default: break
        }
        return nil
    }

    public var description: String {
        switch self {
        case .exercise(let type):  return "通知GPS.\(type)"
        case .camera(let type):    return "通知相机.\(type)"
        case .findPhone(let type): return "通知查找手机.\(type)"
        }
    }
}

// MARK: - 检测记录类型
public enum SRRecordDetectType: CustomStringConvertible {
    case read                   // 开启
    case notify(SRDetectType)   // 心率
    
    var ext: UInt8 {
        switch self {
        case .read: return 0x00
        case .notify(let type): return type.ext
        }
    }
    
    var model: SRModelBase {
        switch self {
        case .read: return SRDetectInfo()
        case .notify(let type): return SRDetectItem(with: type)
        }
    }
    
    static func parse(_ ext: UInt8) -> SRRecordDetectType? {
        if ext == 0 { return .read}
        if let types = SRDetectType.parse(ext) { return .notify(types)}
        return nil
    }
    
    public var description: String {
        switch self {
        case .read:  return "读取检测记录(最近7组)"
        case .notify(let type): return "通知检测记录(\(type))"
        }
    }
}


// MARK: - 详细记录数据组合

/// 记录数据类型
public enum SRRecordType {
    case total      // 总数据
    case step       // 计步
    case heart      // 心率
    case sleep      // 睡眠
    case exercise   // 训练
        
    var value: UInt8 {
        switch self {
        case .total:    return 0x00
        case .step:     return 0x01
        case .heart:    return 0x02
        case .sleep:    return 0x03
        case .exercise: return 0x04
        }
    }
    
    var column: BandRecord.Columns {
        switch self {
        case .total:    return .total
        case .step:     return .step
        case .heart:    return .heart
        case .sleep:    return .sleep
        case .exercise: return .exercise
        }
    }
    
    var model: SRModelBase {
        switch self {
        case .total:    return SRTotal()
        case .step:     return SRStep()
        case .sleep:    return SRSleep()
        case .exercise: return SRExercise()
        case .heart:    return SRHeart()
        }
    }
    
    static var collection: [SRRecordType] = [.total, .step, .heart, .sleep, .exercise]

    static func parse(_ value: UInt8) -> SRRecordType? {
        
        let _value: UInt8 = (value & 0x0F)
        var type: SRRecordType? = nil
        for e in collection { if e.value==_value { type = e; break }}
        return type
    }
    
    public var description: String {
        switch self {
        case .total:    return "total"
        case .step:     return "step"
        case .heart:    return "heart"
        case .sleep:    return "sleep"
        case .exercise: return "exercise"
        }
    }
};

/// 记录数据哪天
public enum SRRecordTn: CustomStringConvertible {
    case t0 // 当天
    case t1
    case t2
    case t3
    case t4
    case t5
    case t6
    case t7
    
    var value: UInt8 {
        switch self {
        case .t0: return 0x00
        case .t1: return 0x10
        case .t2: return 0x20
        case .t3: return 0x30
        case .t4: return 0x40
        case .t5: return 0x50
        case .t6: return 0x60
        case .t7: return 0x70
        }
    }
    
    static var collection: [SRRecordTn] = [.t0, .t1, .t2, .t3, .t4, t5, .t6, .t7]

    static func parse(_ value: UInt8) -> SRRecordTn? {
        let _value: UInt8 = (value & 0xF0)
        var tn: SRRecordTn? = nil
        for e in collection { if e.value==_value { tn = e; break }}
        return tn
    }
    
    var valueFlashListIndex: Int? {
        if self == .t0 { return nil}
        return Int((value >> 4) & 0x0F) - 1
    }
    
    static func parseFlashListIndex(_ idx: Int) -> SRRecordTn? {
        guard idx < collection.count, idx>=0 else { return nil}
        return collection[idx]
    }
    
    public var description: String {
        switch self {
        case .t0: return "t0"
        case .t1: return "t1"
        case .t2: return "t2"
        case .t3: return "t3"
        case .t4: return "t4"
        case .t5: return "t5"
        case .t6: return "t6"
        case .t7: return "t7"
        }
    }
};






// MARK: - ********** Internal **********

enum SRForecastType: CustomStringConvertible {
    case days
    case hours
    var ext: UInt8 {
        switch self {
        case .days:  return 0x00
        case .hours: return 0x01
        }
    }
    
    static func parse(_ ext: UInt8) -> SRForecastType? {
        [days, hours].first(where: { $0.ext == ext })
    }

    var description: String {
        switch self {
        case .days:  return "天气推送.15天"
        case .hours: return "天气推送.24小时"
        }
    }
}

enum SREventType: CustomStringConvertible {
    /// 训练
    public enum Exercise: CustomStringConvertible {
        case on    // 打开
        case off   // 关闭
        var byte1: UInt8 {
            switch self {
            case .on:   return 0x00
            case .off:  return 0x01
            }
        }
        public var description: String {
            switch self {
            case .on:   return "开启"
            case .off:  return "关闭"
            }
        }
    }
    
    /// 相机
    enum Camera: CustomStringConvertible {
        case on   // 打开
        case off  // 关闭
        var ext: UInt8 {
            switch self {
            case .on:   return 0x02
            case .off:  return 0x03
            }
        }
        public var description: String {
            switch self {
            case .on:   return "打开"
            case .off:  return "关闭"
            }
        }
    }
    
    case exercise(Exercise)
    case camera(Camera)
    case findband
    
    var ext: UInt8 {
        switch self {
        case .exercise(_):      return 0x00
        case .camera(let type): return type.ext
        case .findband:         return 0x04
        }
    }
    
    var byte1: UInt8? {
        if case .exercise(let type) = self { return type.byte1 }
        return nil
    }
    
    static func parse(_ ext: UInt8, _ byte1: UInt8) -> SREventType? {
        switch ext {
        case 0x00:
            switch byte1 {
            case 0x00: return .exercise(.on)
            case 0x01: return .exercise(.off)
            default:  break
            }
        case 0x02: return .camera(.on)
        case 0x03: return .camera(.off)
        case 0x04: return .findband
        default: break
        }
        return nil
    }
    
    public var description: String {
        switch self {
        case .exercise(let type):   return "触发事件.Agps(\(type))"
        case .camera(let type):     return "触发事件.Camera(\(type))"
        case .findband:             return "触发事件.FindBand"
        }
    }
}


typealias SRCommandValueType = UInt16
enum SRCommandType {
    case pair
    case time
    case transfer
    case clock
    case user
    case firmware
    case reset
    case drink
    case agps
    case event(SREventType)
    case detect(SRDetectType)
    case weather(SRForecastType)


    case devPair
    case devTime
    case devTransfer
    case devClock
    case devUser
    case devFirmware
    case devDrink
    case devFlashDates

    case devRecord(SRRecordType, SRRecordTn)
    case devNotify(SRNotifyType)
    case devRecordDetect(SRRecordDetectType)

    var cmd: UInt8 { UInt8(truncating: NSNumber(value: value>>8)) }
    var ext: UInt8 { UInt8(truncating: NSNumber(value: value)) }
    var value: SRCommandValueType {
        switch self {
        case .pair:     return 0x0100
        case .time:     return 0x0200
        case .transfer: return 0x0300
        case .clock:    return 0x0400
        case .user:     return 0x0500
        case .firmware: return 0x0600
        case .reset:    return 0x0BFB
        case .drink:    return 0x0D00
        case .agps:     return 0x1000
        case .event(let type):   return (0x0C00 + UInt16(type.ext))
        case .detect(let type):  return (0x1100 + UInt16(type.ext))
        case .weather(let type): return (0x1200 + UInt16(type.ext))

        case .devPair:       return 0x8101
        case .devTime:       return 0x8200
        case .devTransfer:   return 0x8300
        case .devClock:      return 0x8400
        case .devUser:       return 0x8500
        case .devFirmware:   return 0x8600
        case .devFlashDates: return 0x8800
        case .devDrink:      return 0x8D00

        case .devRecord(let type, let tn):   return (0x8900 + UInt16(type.value | tn.value))
        case .devNotify(let type):           return (0x8C00 + UInt16(type.ext))
        case .devRecordDetect(let type):     return (0x9100 + UInt16(type.ext))
        }
    }
    
    var byte1: UInt8? {
        if case .event(let type) = self { return type.byte1 }
        if case .devNotify(let type) = self { return type.byte1 }
        return nil
    }
    
    
    static func parse(_ cmd: UInt8, _ ext: UInt8, _ byte1: UInt8) -> SRCommandType? {
        let value: UInt16 = UInt16(cmd) << 8 + UInt16(ext)
        switch value {
        case pair.value:     return .pair
        case time.value:     return .time
        case transfer.value: return .transfer
        case clock.value:    return .clock
        case user.value:     return .user
        case firmware.value: return .firmware
        case reset.value:    return .reset
        case drink.value:    return .drink
        case agps.value:     return .agps
            
        case devPair.value:       return .devPair
        case devTime.value:       return .devTime
        case devTransfer.value:   return .devTransfer
        case devClock.value:      return .devClock
        case devUser.value:       return .devUser
        case devFirmware.value:   return .devFirmware
        case devFlashDates.value: return .devFlashDates
        case devDrink.value:      return .devDrink
        default: break
        }
        
        /// 触发事件
        if cmd == 0x0C,
           let type = SREventType.parse(ext, byte1) {
            return .event(type)
        }
        /// 手动检测
        if cmd == 0x11,
           let type = SRDetectType.parse(ext) {
            return .detect(type)
        }
        
        /// 天气推送
        if cmd == 0x12,
           let type = SRForecastType.parse(ext) {
            return .weather(type)
        }

        /// 运动、睡眠、心率记录数据
        if cmd == 0x89,
           let type = SRRecordType.parse(ext),
           let tn = SRRecordTn.parse(ext) {
            return .devRecord(type, tn)
        }
        /// 通知事件
        if cmd == 0x8C,
           let type = SRNotifyType.parse(ext, byte1) {
            return .devNotify(type)
        }
        /// 检测记录/ 手动检测通知
        if cmd == 0x91,
           let type = SRRecordDetectType.parse(ext) {
            return .devRecordDetect(type)
        }
        
        return nil
    }
}

extension SRCommandType {
    
    var model: SRModelBase? {
        switch self {
        case .devTime:       return SRDeviceTime()
        case .devClock:      return SRAlarmClock()
        case .devUser:       return SRUserInfo()
        case .devFirmware:   return SRFirmwareInfo()
        case .devDrink:      return SRDrinkInfo()
        case .devFlashDates: return SRFlashDates()
        case .devRecord(let type, _):    return type.model
        case .devRecordDetect(let type): return type.model
        default: return nil
        }
    }
    
    var name: String {
        switch self {
        case .pair:     return "iOS配对"
        case .time:     return "同步时间"
        case .transfer: return "设置传输速度"
        case .clock:    return "保存闹钟信息"
        case .user:     return "保存用户信息"
        case .firmware: return "固件功能"
        case .reset:    return "重置设备"
        case .drink:    return "保存喝水提醒"
        case .agps:     return "Agps运动"
        case .event(let type):   return "\(type)"
        case .detect(let type):  return "\(type)"
        case .weather(let type): return "\(type)"

        case .devPair:       return "读取连接方式"
        case .devTime:       return "读取设备时间"
        case .devTransfer:   return "读取传输速度"
        case .devClock:      return "读取闹钟信息"
        case .devUser:       return "读取用户信息"
        case .devFirmware:   return "读取固件信息"
        case .devFlashDates: return "读取Flash存储日期"
        case .devDrink:      return "读取喝水提醒"

        case .devRecord(let type, let tn):   return "读取记录(\(type), \(tn))"
        case .devNotify(let type):           return "手动测试(\(type))"
        case .devRecordDetect(let type):     return "\(type)"
        }
    }
    
    static func name(_ cmd: UInt8, _ ext: UInt8, _ byte1: UInt8) -> String {
        parse(cmd, ext, byte1)?.name ?? "0x\(Data([cmd, ext]).kitString)"
    }
}

extension SRCommandType: Equatable {
    static func == (lhs: SRCommandType, rhs: SRCommandType) -> Bool {
        (lhs.value == rhs.value && lhs.ext == rhs.ext)
    }
}

// MARK: - 方便判断的简单函数

/// 是否为取写入指令
func isWriteCommand(_ cmd: UInt8) -> Bool{
    return (cmd & 0x80) == 0x80;
}

/// 是否为读取指令
func isReadCommand(_  cmd: UInt8) -> Bool{
    return (cmd & 0x7F) == cmd;
}

///由 包序号 低七位为包编号
func realPacketNo(_ packetNo: UInt8) -> UInt8{
    return (packetNo & 0x7f);
}

/// 判断是否为第一个数据包
func isFirstPacket(_ packetNo: UInt8) -> Bool{
    return ((packetNo & 0x7f) == 0x01);
}

/// 由 包序号最高位判断是否为结束包
func isLastPacket(_ packetNo: UInt8) -> Bool{
    return (packetNo & 0x80) == 0x80;
}

