//
//  SRCommand.swift
//  BandKit
//
//  Created by Mac on 2020/5/4.
//

import Foundation

public enum QueueStyle {
    case push, updatePush
    case insertHeader, updateInsertHeader
    case enforce
}

public class SRCommand {
                
    public var nameText: String { return _nameText}

    public var descripteText: String { return _descripteText}
    
    public func writeToDevice(_ callback: @escaping SRWriteClosure) { _writeToDevice(callback) }
    
    public func writeToDevice() { _writeToDevice(nil) }
    
    public init() { }
    
    // MARK: -  internal
    
    var _queueStyle: QueueStyle = .updatePush
    
    var _overtime: TimeInterval = 5.0
    
    var _commandType: SRCommandType = .pair {
        didSet {
            _nameText = _commandType.name
            _data = Data([_commandType.byte1 ?? 0x00])
            switch _commandType {
            case .pair, .time, .transfer, .clock, .user, .firmware, .reset, .drink, .agps, .detect(_), .weather(_): _queueStyle = .enforce
            default: _queueStyle = .updatePush
            }
        }
    }

    var _cmd: UInt8 { _commandType.cmd }
      
    var _ext: UInt8 { _commandType.ext }
    
    var _byte1: UInt8 { _commandType.byte1 ?? 0x00 }

    var _data: Data = Data([0x00])
    
    var _nameText: String = ""
    
    var _descripteText: String = ""
    
    
    // MARK: -  private

    private func _writeToDevice(_ callback: SRWriteClosure? = nil) {
        var holder : CmdHolder
        if let hasCallback = callback {
            holder = CmdHolder(type: _commandType, data: _data, time: _overtime, reponse: self._reponseClosure(hasCallback))
        } else {
            holder = CmdHolder(type: _commandType, data: _data, time: _overtime)
        }
        if _nameText.count > 0 { holder.describe = _nameText}
        if _descripteText.count > 0 { holder.parseText = _descripteText}
        self._enterToRequest(with: holder)
    }
    
    private func _enterToRequest(with holder: CmdHolder) {
        switch _queueStyle {
        case .push: CmdRequest.push(holder)
        case .insertHeader: CmdRequest.insertHeader(holder)
        case .updatePush: CmdRequest.updatePush(holder)
        case .updateInsertHeader: CmdRequest.updateInsertHeader(holder)
        case .enforce: CmdRequest.enforce(holder)
        }
    }
    
    private func _reponseClosure(_ callback: @escaping SRWriteClosure) -> ReponseClosure {
        let _reponseClosure: ReponseClosure = { (reponse)->Void in
            DispatchQueue.main.async {
                callback(reponse.model, reponse.error)
            }
        }
        return _reponseClosure
    }
}


// MARK: - 链式设置
extension SRCommand {
    public func overtime(_ time: TimeInterval) -> SRCommand {
        _overtime = time
        return self
    }
    public func style(_ style: QueueStyle) -> SRCommand {
        _queueStyle = style
        return self
    }
}


extension SRCommand {
    
    // MARK: - 配对
    public static func readPairWay() -> SRCommand{
        let command = SRCommand()
        command._commandType = .devPair
        command._overtime = 2.0
        return command
    }

    public static func pairiOS() -> SRCommand{
        let data = Data([0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00])
        let command = SRCommand()
        command._commandType = .pair
        command._data = data
        command._overtime = 0
        return command
    }
    
    
    // MARK: - 时间
    
    public static func readTime() -> SRCommand{
        let command = SRCommand()
        command._commandType = .devTime
        return command
    }
    
    public static func syncTime(_ deviceTime: SRDeviceTime) -> SRCommand{
        let command = SRCommand()
        command._commandType = .time
        command._data = deviceTime.data()
        command._descripteText = String(describing: deviceTime)
        return command
    }
    
    // MARK: - 传输速度
    
    public static func readTransferSpeed() -> SRCommand{
        let command = SRCommand()
        command._commandType = .devTransfer
        return command
    }
    
    public static func transferSpeed(_ type: SRTransferSpeedType) -> SRCommand{
        let data = Data([type.value])
        let command = SRCommand()
        command._commandType = .transfer
        command._data = data
        command._nameText += type.text
        return command
    }

    
    // MARK: - 闹钟

    public static func readAlarmClock() -> SRCommand{
        let command = SRCommand()
        command._commandType = .devClock
        return command
    }
    
    public static func setAlarmClock(_ alarmClock: SRAlarmClock) -> SRCommand{
        let command = SRCommand()
        command._commandType = .clock
        command._data = alarmClock.data()
        command._descripteText = String(describing: alarmClock)
        command._overtime = 10
        return command
    }
    
    // MARK: - 用户信息

    public static func readUserInfo() -> SRCommand{
        let command = SRCommand()
        command._commandType = .devUser
        return command
    }
    
    public static func setUserInfo(_ userInfo: SRUserInfo) -> SRCommand{
        let command = SRCommand()
        command._commandType = .user
        command._data = userInfo.data()
        command._descripteText = String(describing: userInfo)
        command._overtime = 10
        return command
    }
    
    // MARK: - 固件信息
    public static func readFirmwareInfo() -> SRCommand{
        let command = SRCommand()
        command._commandType = .devFirmware
        return command
    }
    
    
    public static func firmwareControl(type: SRFirmwareControlType) -> SRCommand{
        let command = SRCommand()
        command._commandType = .firmware
        command._data = type.data
        command._nameText += type.name
        command._overtime = 3
        return command
    }

        
    // MARK: - 恢复出厂
    
    public static func restFactory() -> SRCommand{
        let command = SRCommand()
        command._commandType = .reset
        command._overtime = 0
        return command
    }
    
    // MARK: - Event (无应答)
 
    public static func findBandEvent() -> SRCommand{
        let command = SRCommand()
        command._commandType = .event(.findband)
        command._overtime = 0
        return command
    }
    
    /// type  0  进入、1 退出
    public static func cameraEvent(_ type: Int) -> SRCommand{
        let type: SREventType.Camera = (type == 0) ? .on : .off
        let command = SRCommand()
        command._commandType = .event(.camera(type))
        command._overtime = 0
        return command
    }
    
    /// state  0  开启、1 结束
    public static func gpsEvent(_ state: Int) -> SRCommand{
        let type: SREventType.Exercise = (state == 0) ? .on : .off
        let command = SRCommand()
        command._commandType = .event(.exercise(type))
        command._overtime = 0
        return command
    }
    
    // MARK: - Flash 日期列表

    public static func readFlashDates() -> SRCommand{
        let command = SRCommand()
        command._commandType = .devFlashDates
        return command
    }
    
    // MARK: - Flash 详细记录

    public static func readRecordDetail(_ type: SRRecordType, _ tn: SRRecordTn) -> SRCommand{
        
        let command = SRCommand()
        command._commandType = .devRecord(type, tn)
        return command
    }
    
    // MARK: - 读取喝水提醒

    public static func readDrinkInfo() -> SRCommand{
        let command = SRCommand()
        command._commandType = .devDrink
        return command
    }
    
    public static func setDrinkInfo(_ drinkInfo: SRDrinkInfo) -> SRCommand{
        let command = SRCommand()
        command._commandType = .drink
        command._descripteText = String(describing: drinkInfo)
        command._data = drinkInfo.data()
        command._overtime = 10
        return command
    }
    
    // MARK - Gpx 运动
    public static func exerciseStart(_ type: SRSportType) -> SRCommand{
        let data = Data([type.value])
        let command = SRCommand()
        command._commandType = .agps
        command._data = data
        command._overtime = 3
        return command
    }
    
    public static func exerciseEnd() -> SRCommand{
        let data = Data([0x01])
        let command = SRCommand()
        command._commandType = .agps
        command._data = data
        command._overtime = 3
        return command
    }
    
    
    // MARK: - 检测 （血压/心率）
    
    public static func readDetectRecords() -> SRCommand{
        let command = SRCommand()
        command._commandType = .devRecordDetect(.read)
        return command
    }
    
    public static func detect(_ type: SRDetectType = .bpm) -> SRCommand{
        let command = SRCommand()
        command._commandType = .detect(type)
        return command
    }
    
    // MARK: - 天气推送
    public static func weatherForecastHour(_ obj: SRWeatherForecastHour? = SRWeatherForecastHour.forecastHour) -> SRCommand?{
        guard let hasObj = obj, let data = hasObj.data() else { return nil }
        let command = SRCommand()
        command._commandType = .weather(.hours)
        command._data = data
        command._overtime = 5
        command._descripteText = String(describing: hasObj)
        return command
    }
    
    public static func weatherForecastDay(_ obj: SRWeatherForecastDay? = SRWeatherForecastDay.forecastDay) -> SRCommand?{
       guard let hasObj = obj, let data = hasObj.data() else { return nil }
        let command = SRCommand()
        command._commandType = .weather(.days)
        command._data = data
        command._overtime = 5
        command._descripteText = String(describing: hasObj)
        return command
    }
}
