//
//  BandRecord.swift
//  CTFit
//
//  Created by Mac on 2020/6/19.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation
import GRDB

public struct BandRecord: Codable {
    var id: Int?
    /// 日期：（2020-06-23）
    var date: String?
    /// 数据字段标志位
    var flag: Int?
    /// 总数据
    var total: Data?
    /// 详细计步数据
    var step: Data?
    /// 静态数据
    var heart: Data?
    /// 睡眠数据
    var sleep: Data?
    /// 运动训练数据
    var exercise: Data?
}

// MARK: - ColumnExpression
extension BandRecord {
    enum Columns: String, ColumnExpression {
        case id, date, flag
        case total      = "SportTotal"
        case step       = "SportWalk"
        case heart      = "SportHeart"
        case sleep      = "SportSleep"
        case exercise   = "SportExercise"
        
        var mask: Int {
            switch self {
            case .total:    return 0x01
            case .step:     return 0x02
            case .heart:    return 0x04
            case .sleep:    return 0x08
            case .exercise: return 0x10
            default:
                return 0x00
            }
        }        
        static let maskAll: Int = 31
    }
}

// MARK: - TableRecord
/// 使用 TableRecord 协议, 告知操作数据表
extension BandRecord: TableRecord {
    //  数据表名字 如果不自定义 将使用默认的表名, 默认为小写开头驼峰命名 例如 UserEntity -> userEntity
    public static var databaseTableName: String { return "BHSportData" }
    // 声明列名表示的方式
    public static var databaseSelection: [SQLSelectable] = [Columns.id, Columns.date, Columns.flag,
                                                     Columns.total, Columns.step, Columns.heart, Columns.sleep, Columns.exercise]
}

// MARK: - FetchableRecord
/// FetchableRecord 进行查询操作
/// 使用了Codable可以不实现  init (row: Row)
/// 未使用 Codable = Decodable & Encodable 协议的需要实现 init (row: Row)
extension BandRecord: FetchableRecord {
    public init(row: Row) {
        id = row[Columns.id]
        date = row[Columns.date]
        flag = row[Columns.flag]
        total = row[Columns.total]
        step = row[Columns.step]
        heart = row[Columns.heart]
        sleep = row[Columns.sleep]
        exercise = row[Columns.exercise]
    }
}

// MARK: - MutablePersistableRecord
/// 使用PersistableRecord / MutablePersistableRecord插入更新保存数据,
/// 存储模型是Class使用PersistableRecord,
/// 存储模型是struct使用MutablePersistableRecord.
/// 两者区别在于 MutablePersistableRecord save() insert() 是 mutating.
extension BandRecord: MutablePersistableRecord {
    
    public func encode(to container: inout PersistenceContainer) {
        container[Columns.id] = id
        container[Columns.date] = date
        container[Columns.flag] = flag
        container[Columns.total] = total
        container[Columns.step] = step
        container[Columns.heart] = heart
        container[Columns.sleep] = sleep
        container[Columns.exercise] = exercise
    }
    
    mutating public func didInsert(with rowID: Int64, for column: String?) {
    }
}

extension BandRecord {
    
    /// 获取数据库对象
    private static var dbQueue: DatabaseQueue? {
        SQLiteManager.dbQueue
    }
    
    /// 创建数据库
    private static func createTable() -> Void {
        guard let _dbQueue = dbQueue else { return }
        try! _dbQueue.inDatabase { (db) -> Void in
            /// 判断是否存在数据库
            if try db.tableExists(databaseTableName) { return }
            
            /// 创建数据库表
            try db.create(table: databaseTableName, temporary: false, ifNotExists: true, body: { (t) in
                t.autoIncrementedPrimaryKey("id")
                t.column(Columns.date.rawValue, .text)
                t.column(Columns.flag.rawValue, .integer)
                t.column(Columns.total.rawValue, .blob)
                t.column(Columns.step.rawValue,.blob)
                t.column(Columns.heart.rawValue, .blob)
                t.column(Columns.sleep.rawValue, .blob)
                t.column(Columns.exercise.rawValue, .blob)
            })
        }
    }
}


// MARK: - 插入&更新

extension BandRecord {
        
    /// 更新
    private static func update(record: BandRecord, _ column: Columns, _ value: DatabaseValueConvertible?, isToady: Bool = true) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        /// 事务 更新场景
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                if !isToady { temp.flag = (temp.flag ?? 0) | column.mask }
                if column == .flag      { temp.flag = value as? Int }
                if column == .total     { temp.total = value as? Data }
                if column == .step      { temp.step  = value as? Data }
                if column == .heart     { temp.heart = value as? Data }
                if column == .sleep     { temp.sleep = value as? Data }
                if column == .exercise  { temp.exercise = value as? Data }
                try temp.update(db)
                return Database.TransactionCompletion.commit
            } catch {
                BandLog.e("\(error)")
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 插入
    private static func insert( _ dateStr: String, _ column: Columns, _ value: DatabaseValueConvertible?, isToady: Bool = true) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        var record = BandRecord(date: dateStr)
        if !isToady { record.flag = (record.flag ?? 0) | column.mask }
        if column == .flag      { record.flag = value as? Int }
        if column == .total     { record.total = value as? Data }
        if column == .step      { record.step  = value as? Data }
        if column == .heart     { record.heart = value as? Data }
        if column == .sleep     { record.sleep = value as? Data }
        if column == .exercise  { record.sleep = value as? Data }
        
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                try temp.insert(db)
                return Database.TransactionCompletion.commit
            } catch {
                BandLog.e("\(error)")
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 保存单个数据， dateStr 相等则更新， 否则插入
    internal static func save( _ dateStr: String, _ column: Columns, _ value: DatabaseValueConvertible?, isToady: Bool = true) {
        guard dateStr > "2010" else { return }
        /// 判断是否存在
        if let record = queryRecord(dateStr:dateStr) {
            self.update(record: record, column, value, isToady: isToady)
        } else {
            self.insert(dateStr, column, value, isToady: isToady)
        }
    }
    
    /// 清楚 flag 和 记录字段
    internal static func clear(dateStr: String) {
        guard let _dbQueue = dbQueue else { return }
        guard var record = queryRecord(dateStr:dateStr) else { return }
        /// 创建数据库表
        self.createTable()
        /// 事务 更新场景
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try record.updateChanges(db) { (record) in
                    record.flag = 0
                    record.total = nil
                    record.step = nil
                    record.heart = nil
                    record.sleep = nil
                    record.exercise = nil
                }
                return Database.TransactionCompletion.commit
            } catch {
                BandLog.e("\(error)")
                return Database.TransactionCompletion.rollback
            }
        }
    }
}


// MARK: - 删除
extension BandRecord {
    /// 根据 dateStr 删除
    private static func delete(dateStr: String) {
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try BandRecord.filter(Column(Columns.date.rawValue) == dateStr).deleteAll(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    ///: 移除表
    internal static func remove() {
        guard let _dbQueue = dbQueue else { return }
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try db.drop(table: databaseTableName)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    ///: - 删除所有
    internal static func deleteAll() {
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try BandRecord.deleteAll(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
}

// MARK: - 查询

extension BandRecord {
        
    private static let formatter :DateFormatter = {
        let fm = DateFormatter();
        fm.dateFormat = "yyyy-MM-dd";
        return fm;
    }()
    
    private static func date(_ dateStr: String) -> Date? {
        return formatter.date(from: dateStr)
    }
    
    /// 查询某天的记录
    private static func queryRecord(date: Date) -> BandRecord? {
        return queryRecord(dateStr: formatter.string(from: date))
    }
    
    /// 查询某天的记录
    private static func queryRecord(dateStr: String) -> BandRecord? {
        guard let _dbQueue = dbQueue else { return nil }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> BandRecord? in
            return try BandRecord.filter(Column(Columns.date.rawValue) == dateStr).fetchOne(db)
        })
    }
    
    /// 查询时间段内的记录
    private static func queryRecord(from fromDateStr: String, to toDateStr: String) -> [BandRecord] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [BandRecord] in
            return try BandRecord.filter(Column(Columns.date.rawValue) >= fromDateStr &&
                Column(Columns.date.rawValue) <= toDateStr).order(Columns.date).fetchAll(db)
        })
    }
    
    /// 查询所有
    private static func queryAllRecord() -> [BandRecord] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [BandRecord] in
            return try BandRecord.order(Columns.date).fetchAll(db)
        })
    }
    
    // MAR: - Internal
    
    internal static func dateStr(_ date: Date) -> String {
        return formatter.string(from: date)
    }
    
    /// 查询 Flag
    internal static func queryFlag(_ dateStr: String) -> Int {
        guard let flag = queryColumn(dateStr, .flag) as? Int else { return 0 }
        return flag
    }
    
    /// 查询字段
    internal static func queryColumn(_ dateStr: String, _ column: Columns) -> Any? {
        guard let record = queryRecord(dateStr: dateStr) else { return nil }
        if column == .date      { return record.date }
        if column == .flag      { return record.flag }
        if column == .total     { return record.total }
        if column == .step      { return record.step }
        if column == .heart     { return record.heart }
        if column == .sleep     { return record.sleep }
        if column == .exercise  { return record.exercise }
        return nil
    }
    
    /// 查询所有, 指定列
    internal static func queryAllRecord(column: Columns) -> [Any] {
        var items = [Any]()
        let records = queryAllRecord()
        records.forEach { (record) in
            if column == .date, let item = record.date { items.append(item) }
            if column == .flag, let item = record.flag { items.append(item) }
            if column == .total, let item = record.total { items.append(item) }
            if column == .step, let item = record.step { items.append(item) }
            if column == .heart, let item = record.heart { items.append(item) }
            if column == .sleep, let item = record.sleep { items.append(item) }
            if column == .exercise, let item = record.exercise { items.append(item) }
        }
        return items
    }
    
    /// 查询时间段内的记录
    internal static func queryRecord(from fromDateStr: String, to toDateStr: String, column: Columns) -> [Any] {
        var items = [Any]()
        let records = queryRecord(from: fromDateStr, to: toDateStr)
        records.forEach { (record) in
            if column == .date, let item = record.date { items.append(item) }
            if column == .flag, let item = record.flag { items.append(item) }
            if column == .total, let item = record.total { items.append(item) }
            if column == .step, let item = record.step { items.append(item) }
            if column == .heart, let item = record.heart { items.append(item) }
            if column == .sleep, let item = record.sleep { items.append(item) }
            if column == .exercise, let item = record.exercise { items.append(item) }
        }
        return items
    }
}
    
extension BandRecord {
    
    /// 运动详情页数据查询
    public typealias StepDetailDayClosure = (_ total: SRTotal?, _ step: SRStep?, _ exercise:SRExercise?) -> Void
    public static func query(with date: Date, stepDetailDay: @escaping StepDetailDayClosure) {
        let tuples = queryStepDetail(with: date)
        stepDetailDay(tuples.0, tuples.1, tuples.2)
    }
    public static func queryStepDetail(with date: Date) -> (SRTotal?, SRStep?, SRExercise?) {
        let dateStr = self.dateStr(date)
        guard let record = queryRecord(dateStr: dateStr) else { return (nil, nil, nil) }
        var total: SRTotal? = nil
        var step: SRStep? = nil
        var exercise: SRExercise? = nil
        if let data = record.total { total = SRTotal(); total!.parse(data) }
        if let data = record.step { step = SRStep(); step!.parse(data) }
        if let data = record.exercise { exercise = SRExercise(); exercise!.parse(data) }
        return (total, step, exercise)
    }
    
    /// 运动统计页数据查询
    public typealias StepDetailIntervalClosure = (_ totalArray: [SRTotal], _ exerciseArray: [SRExercise]) -> Void
    public static func query(frome start: Date, to end: Date, stepIntervalDay: @escaping StepDetailIntervalClosure) {
        let tuples = queryStepInterval(frome: start, to: end)
        stepIntervalDay(tuples.0, tuples.1)

    }
    public static func queryStepInterval(frome start: Date, to end: Date) -> ([SRTotal], [SRExercise]) {
        let fromDateStr = self.dateStr(start)
        let toDateStr = self.dateStr(end)
        var totalArray = [SRTotal]()
        var exerciseArray = [SRExercise]()
        let records = queryRecord(from: fromDateStr, to: toDateStr)
        records.forEach { (record) in
            if let data = record.total { let total = SRTotal(); total.parse(data); totalArray.append(total) }
            if let data = record.exercise { let exercise = SRExercise(); exercise.parse(data);  exerciseArray.append(exercise) }
        }
        return(totalArray, exerciseArray)
    }

    
    /// 导出CSV文件记录数据查询
    public typealias CSVClosure = (_ totalArray: [SRTotal], _ stepArray: [SRStep], _ heartArray: [SRHeart], _ sleepArray: [SRSleep]) -> Void
    public static func csvQueryAll(_ closure: @escaping CSVClosure) {
        var totalArray = [SRTotal]()
        var stepArray = [SRStep]()
        var heartArray = [SRHeart]()
        var sleepArray = [SRSleep]()
        let records = queryAllRecord()
        records.reversed().forEach { (record) in
            if let data = record.total { let total = SRTotal(); total.parse(data); totalArray.append(total) }
            if let data = record.step { let step = SRStep(); step.parse(data);  stepArray.append(step) }
            if let data = record.heart { let heart = SRHeart(); heart.parse(data);  heartArray.append(heart) }
            if let data = record.sleep { let sleep = SRSleep(); sleep.parse(data);  sleepArray.append(sleep) }
        }
        closure(totalArray, stepArray, heartArray, sleepArray)
    }
}

