//
//  DetectRecord.swift
//  BandKit
//
//  Created by Mac on 2020/10/28.
//

import Foundation
import GRDB

struct DetectRecord: Codable {
    var id: Int?
    /// 类型
    var type: UInt8
    /// 时间：（2020-06-23 08:01）
    var time: Double
    //// 心率 或者 血压高压值 (收缩压)
    var heartorsbp: UInt8
    /// /// 血氧 或者 血压低压值 (舒张压)
    var spo2ordbp: UInt8
}

// MARK: - ColumnExpression
extension DetectRecord {
    enum Columns: String, ColumnExpression {
        case id, type, time, heartorsbp, spo2ordbp
    }
}

// MARK: - TableRecord
/// 使用 TableRecord 协议, 告知操作数据表
extension DetectRecord: TableRecord {
    //  数据表名字 如果不自定义 将使用默认的表名, 默认为小写开头驼峰命名 例如 UserEntity -> userEntity
    public static var databaseTableName: String { return "DetectRecord" }
    // 声明列名表示的方式
    public static var databaseSelection: [SQLSelectable] = [Columns.id, Columns.type, Columns.time,
                                                     Columns.heartorsbp, Columns.spo2ordbp]
}

// MARK: - FetchableRecord
/// FetchableRecord 进行查询操作
/// 使用了Codable可以不实现  init (row: Row)
/// 未使用 Codable = Decodable & Encodable 协议的需要实现 init (row: Row)
extension DetectRecord: FetchableRecord {
    public init(row: Row) {
        id = row[Columns.id]
        type = row[Columns.type]
        time = row[Columns.time]
        heartorsbp = row[Columns.heartorsbp]
        spo2ordbp = row[Columns.spo2ordbp]
    }
}

// MARK: - MutablePersistableRecord
/// 使用PersistableRecord / MutablePersistableRecord插入更新保存数据,
/// 存储模型是Class使用PersistableRecord,
/// 存储模型是struct使用MutablePersistableRecord.
/// 两者区别在于 MutablePersistableRecord save() insert() 是 mutating.
extension DetectRecord: MutablePersistableRecord {
    
    public func encode(to container: inout PersistenceContainer) {
        container[Columns.id] = id
        container[Columns.type] = type
        container[Columns.time] = time
        container[Columns.heartorsbp] = heartorsbp
        container[Columns.spo2ordbp] = spo2ordbp
    }
    
    mutating public func didInsert(with rowID: Int64, for column: String?) {
        
    }
    
}

extension DetectRecord {
    
    /// 获取数据库对象
    private static var dbQueue: DatabaseQueue? {
        SQLiteManager.dbQueue
    }
    /// 创建数据库
    private static func createTable() -> Void {
        guard let _dbQueue = dbQueue else { return }
        try! _dbQueue.inDatabase { (db) -> Void in
            /// 判断是否存在数据库
            if try db.tableExists(databaseTableName) { return }
            
            /// 创建数据库表
            try db.create(table: databaseTableName, temporary: false, ifNotExists: true, body: { (t) in
                t.autoIncrementedPrimaryKey(Columns.id.rawValue)
                t.column(Columns.type.rawValue, .integer)
                t.column(Columns.time.rawValue, .double)
                t.column(Columns.heartorsbp.rawValue, .integer)
                t.column(Columns.spo2ordbp.rawValue,.integer)
            })
        }
    }
}


// MARK: - 插入&更新

extension DetectRecord {
    
    private static var now: Date { Date() }
    
    private static var latestDictionray: [SRDetectType: DetectRecord] = [:]
    
    private static func latestItem(type: SRDetectType) -> DetectRecord? {
        if let item = latestDictionray[type] { return item }
        if let latestItem =  queryRecordLatest(type: type) {
            latestDictionray[type] = latestItem;
            return latestItem
        }
        return nil
    }
    
    
    
    /// 更新
    private static func update(_ record: DetectRecord, _ item: SRDetectItem) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        /// 事务 更新场景
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                temp.heartorsbp = item.heartorsbp
                temp.spo2ordbp = item.spo2ordbp
                try temp.update(db)
                return Database.TransactionCompletion.commit
            } catch {
                BandLog.e("\(error)")
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 插入
    private static func insert(_ item: SRDetectItem) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        
        let record = DetectRecord(type: item.type.value, time: item.time, heartorsbp: item.heartorsbp, spo2ordbp: item.spo2ordbp)
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                try temp.insert(db)
                return Database.TransactionCompletion.commit
            } catch {
                BandLog.e("\(error)")
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 保存单个数据， dateStr 相等则更新， 否则插入
    private static func save(with item: SRDetectItem) -> Bool {
        guard item.time > SRCalendarUtils.date(from: "2010", format: "yyyy").timeIntervalSince1970 else { return false}
        guard item.time < now.timeIntervalSince1970 + 10 else { return false}
        // if let latestItem =  latestItem(type: item.type),  item.time < latestItem.time { return false}

        /// 判断是否存在
        if let record = queryRecord(item) {
            self.update(record, item)
            if item.type == .bpm {  _ = SRDetectItem.heartRateItems.popLast() }
            if item.type == .bp  { _ = SRDetectItem.bloodPressureItems.popLast() }
        } else {
            self.insert(item)
        }
        
        if item.type == .bpm { SRDetectItem.heartRateItems.append(item) }
        if item.type == .bp  { SRDetectItem.bloodPressureItems.append(item) }
        return true
    }
}


// MARK: - 删除
extension DetectRecord {
    
    /// 删除所有记录
    internal static func deleteAll() {
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try DetectRecord.deleteAll(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 移除表
    internal static func remove() {
        guard let _dbQueue = dbQueue else { return }
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try db.drop(table: databaseTableName)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
}

// MARK: - 查询

extension DetectRecord {
        
    private static let formatter :DateFormatter = {
        let fm = DateFormatter();
        fm.dateFormat = "yyyy-MM-dd HH:mm";
        return fm;
    }()
    
    private static func dateStr(_ date: Date) -> String {
        return formatter.string(from: date)
    }
    
    private static func date(_ dateStr: String) -> Date? {
        return formatter.date(from: dateStr)
    }
    
    /// 查询
    private static func queryRecord( _ item: SRDetectItem) -> DetectRecord? {
        guard let _dbQueue = dbQueue else { return nil }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> DetectRecord? in
            return try DetectRecord.filter(Column(Columns.type.rawValue)==item.type.value &&
                                            Column(Columns.time.rawValue)==item.time ).fetchOne(db)
        })
    }
        
    /// 查询所有
    private static func queryAllRecord() -> [DetectRecord] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [DetectRecord] in
            return try DetectRecord.order(Columns.time).fetchAll(db)
        })
    }
    
    /// 查询所有
    private static func queryAllRecord(type: SRDetectType, limit: Int = Int.max) -> [DetectRecord] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [DetectRecord] in
            return try DetectRecord.filter(Column(Columns.type.rawValue)==type.value)
                .order(Columns.time).reversed().limit(limit)
                .fetchAll(db)
        })
    }
    
    /// 查询最近的记录
    private static func queryRecordLatest(type: SRDetectType) -> DetectRecord? {
        let records = queryAllRecord(type: type, limit: 1)
        return records.first
    }
}

extension DetectRecord {
    /// 查询所有记录
    private static func old_queryAllRecord(type: SRDetectType, limit: Int = Int.max) -> [DetectRecord] {
        switch type {
        case .bpm:
            return DetectRecordHeart.queryAllRecord(limit: limit).map { (e) -> DetectRecord in
                DetectRecord(type: type.value, time: e.BG_time, heartorsbp: e.BG_heartorsbp, spo2ordbp: e.BG_spo2ordbp)
            }
        case .bp:
            return DetectRecordBloodPressure.queryAllRecord(limit: limit).map { (e) -> DetectRecord in
                DetectRecord(type: type.value, time: e.BG_time, heartorsbp: e.BG_heartorsbp, spo2ordbp: e.BG_spo2ordbp)
            }
        }
    }
    
    /// 保存记录
    private static func old_save(with item: SRDetectItem) -> Bool {
        switch item.type {
        case .bpm: return DetectRecordHeart.save(with: item)
        case .bp:  return DetectRecordBloodPressure.save(with: item)
        }
    }
}

extension DetectRecord {
    /// 查询所有记录
    static func queryAllRecord(type: SRDetectType, limit: Int = Int.max, old: Bool = false) -> [DetectRecord] {
        if old { return old_queryAllRecord(type: type, limit: limit) }
        return queryAllRecord(type: type, limit: limit)
    }
    
    /// 保存记录
    static func save(with item: SRDetectItem, old: Bool = false) -> Bool {
        if old { return old_save(with: item) }
        return save(with: item)
    }
}
