//
//  DetectRecordBloodPressure.swift
//  BandKit
//
//  Created by Mac on 2020/12/16.
//

import Foundation
import GRDB

struct DetectRecordBloodPressure: Codable {
    var BG_bg_id: Int?
    /// 时间：（2020-06-23 08:01）
    var BG_time: Double = 0
    //// 心率 或者 血压高压值 (收缩压)
    var BG_heartorsbp: UInt8 = 0
    /// /// 血氧 或者 血压低压值 (舒张压)
    var BG_spo2ordbp: UInt8 = 0
    
    var BG_bg_createTime: String = ""
    var BG_bg_updateTime: String = ""
    var BG_year: UInt16 = 0
    var BG_month: UInt8 = 0
    var BG_day: UInt8 = 0
    var BG_hour: UInt8 = 0
    var BG_minute: UInt8 = 0
    
    
    init(with detectItem: SRDetectItem, isUpdate:Bool = false) {
        self.BG_time = detectItem.time
        self.BG_heartorsbp = detectItem.heartorsbp
        self.BG_spo2ordbp = detectItem.spo2ordbp
        let nowStr = DetectRecordBloodPressure.formatter.string(from: Date())
        if !isUpdate { self.BG_bg_createTime = nowStr }
        self.BG_bg_updateTime = nowStr
        
        let date = Date(timeIntervalSince1970: BG_time)
        let calendar = Calendar(identifier: .gregorian)
        let unitFlags: Set<Calendar.Component> = [.year, .month, .day, .hour, .minute]
        let components =  calendar.dateComponents(unitFlags, from: date)
        self.BG_year    = UInt16(components.year!)
        self.BG_month   = UInt8(components.month!)
        self.BG_day     = UInt8(components.day!)
        self.BG_hour    = UInt8(components.hour!)
        self.BG_minute  = UInt8(components.minute!)
    }
}

// MARK: - ColumnExpression
extension DetectRecordBloodPressure {
    enum Columns: String, ColumnExpression {
        case BG_bg_id, BG_time, BG_heartorsbp, BG_spo2ordbp
        case BG_bg_createTime, BG_bg_updateTime
        case BG_year, BG_month, BG_day
        case BG_hour, BG_minute
    }
    
}

// MARK: - TableRecord
/// 使用 TableRecord 协议, 告知操作数据表
extension DetectRecordBloodPressure: TableRecord {
    //  数据表名字 如果不自定义 将使用默认的表名, 默认为小写开头驼峰命名 例如 UserEntity -> userEntity
    public static var databaseTableName: String { return "BHDetectBloodModel" }
    // 声明列名表示的方式
    public static var databaseSelection: [SQLSelectable] = [Columns.BG_bg_id,
                                                            Columns.BG_time, Columns.BG_heartorsbp, Columns.BG_spo2ordbp,
                                                            Columns.BG_bg_createTime,Columns.BG_bg_updateTime,
                                                            Columns.BG_year, Columns.BG_month, Columns.BG_day,
                                                            Columns.BG_hour, Columns.BG_minute,
                                                            ]
}

// MARK: - FetchableRecord
/// FetchableRecord 进行查询操作
/// 使用了Codable可以不实现  init (row: Row)
/// 未使用 Codable = Decodable & Encodable 协议的需要实现 init (row: Row)
extension DetectRecordBloodPressure: FetchableRecord {
    public init(row: Row) {
        BG_bg_id = row[Columns.BG_bg_id]
        BG_time = row[Columns.BG_time]
        BG_heartorsbp = row[Columns.BG_heartorsbp]
        BG_spo2ordbp = row[Columns.BG_spo2ordbp]
        
        BG_bg_createTime = row[Columns.BG_bg_createTime]
        BG_bg_updateTime = row[Columns.BG_bg_updateTime]
        BG_year = row[Columns.BG_year]
        BG_month = row[Columns.BG_month]
        BG_day = row[Columns.BG_day]
        BG_hour = row[Columns.BG_hour]
        BG_minute = row[Columns.BG_minute]

    }
}

// MARK: - MutablePersistableRecord
/// 使用PersistableRecord / MutablePersistableRecord插入更新保存数据,
/// 存储模型是Class使用PersistableRecord,
/// 存储模型是struct使用MutablePersistableRecord.
/// 两者区别在于 MutablePersistableRecord save() insert() 是 mutating.
extension DetectRecordBloodPressure: MutablePersistableRecord {
    
    public func encode(to container: inout PersistenceContainer) {
        container[Columns.BG_bg_id] = BG_bg_id
        container[Columns.BG_time] = BG_time
        container[Columns.BG_heartorsbp] = BG_heartorsbp
        container[Columns.BG_spo2ordbp] = BG_spo2ordbp
        
        container[Columns.BG_bg_createTime] = BG_bg_createTime
        container[Columns.BG_bg_updateTime] = BG_bg_updateTime
        container[Columns.BG_year] = BG_year
        container[Columns.BG_month] = BG_month
        container[Columns.BG_day] = BG_day
        container[Columns.BG_hour] = BG_hour
        container[Columns.BG_minute] = BG_minute
    }
    
    mutating public func didInsert(with rowID: Int64, for column: String?) {
        
    }
    
}

extension DetectRecordBloodPressure {
    
    /// 获取数据库对象
    private static var dbQueue: DatabaseQueue? {
        SQLiteManager.dbQueue_s
    }
    
    /// 创建数据库
    private static func createTable() -> Void {
        guard let _dbQueue = dbQueue else { return }
        try! _dbQueue.inDatabase { (db) -> Void in
            /// 判断是否存在数据库
            if try db.tableExists(databaseTableName) { return }
            
            /// 创建数据库表
            try db.create(table: databaseTableName, temporary: false, ifNotExists: true, body: { (t) in
                t.autoIncrementedPrimaryKey(Columns.BG_bg_id.rawValue)
                t.column(Columns.BG_heartorsbp.rawValue, .integer)
                t.column(Columns.BG_spo2ordbp.rawValue,.integer)
                t.column(Columns.BG_time.rawValue, .double)
                t.column(Columns.BG_bg_createTime.rawValue, .text)
                t.column(Columns.BG_bg_updateTime.rawValue, .text)
                t.column(Columns.BG_year.rawValue,.integer)
                t.column(Columns.BG_month.rawValue,.integer)
                t.column(Columns.BG_day.rawValue,.integer)
                t.column(Columns.BG_hour.rawValue,.integer)
                t.column(Columns.BG_minute.rawValue,.integer)
            })
        }
    }
}

// MARK: - 查询
extension DetectRecordBloodPressure {
    
    static let formatter :DateFormatter = {
        let fm = DateFormatter();
        fm.dateFormat = "yyyy-MM-dd HH:mm:ss.SSS";
        return fm;
    }()
    
    /// 查询
    private static func queryRecord( _ item: SRDetectItem) -> DetectRecordBloodPressure? {
        guard let _dbQueue = dbQueue else { return nil }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead { (db) -> DetectRecordBloodPressure? in
            return try DetectRecordBloodPressure.filter(Column(Columns.BG_time.rawValue)==item.time).fetchOne(db)
        }
       
    }
    
    /// 查询所有
    static func queryAllRecord(limit: Int = Int.max) -> [DetectRecordBloodPressure] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        return try! _dbQueue.unsafeRead({ (db) -> [DetectRecordBloodPressure] in
            return try DetectRecordBloodPressure.order(Columns.BG_time).limit(limit).fetchAll(db)
        })
    }
}

// MARK: - 插入&更新

extension DetectRecordBloodPressure {
    
    private static var now: Date { Date() }
    
    private static var _latestRecord: DetectRecordBloodPressure?
    
    private static var latestRecord: DetectRecordBloodPressure? {
        if let item = _latestRecord { return item }
        if let record =  queryRecordLatest() {
            _latestRecord = record;
            return record
        }
        return nil
    }
    
    /// 查询
    private static func queryRecordLatest() -> DetectRecordBloodPressure? {
        let records = queryAllRecord(limit: 1)
        return records.first
    }
    
    /// 更新
    private static func update(_ record: DetectRecordBloodPressure, _ item: SRDetectItem) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        /// 事务 更新场景
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                temp.BG_heartorsbp = item.heartorsbp
                temp.BG_spo2ordbp = item.spo2ordbp
                temp.BG_bg_updateTime = formatter.string(from: now)
                try temp.update(db)
                return Database.TransactionCompletion.commit
            } catch {
                BandLog.e("\(error)")
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 插入
    private static func insert(_ item: SRDetectItem) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        let record = DetectRecordBloodPressure(with: item, isUpdate: false)
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                var temp = record
                try temp.insert(db)
                return Database.TransactionCompletion.commit
            } catch {
                BandLog.e("\(error)")
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 保存单个数据， dateStr 相等则更新， 否则插入
    static func save(with item: SRDetectItem) -> Bool {
        guard item.time > SRCalendarUtils.date(from: "2010", format: "yyyy").timeIntervalSince1970 else { return false}
        guard item.time < now.timeIntervalSince1970 + 10 else { return false}
        if let record =  latestRecord,  item.time < record.BG_time { return false}

        /// 判断是否存在
        if let record = queryRecord(item) {
            self.update(record, item)
            if item.type == .bpm {  _ = SRDetectItem.heartRateItems.popLast() }
            if item.type == .bp  { _ = SRDetectItem.bloodPressureItems.popLast() }
        } else {
            self.insert(item)
        }
        
        if item.type == .bpm { SRDetectItem.heartRateItems.append(item) }
        if item.type == .bp  { SRDetectItem.bloodPressureItems.append(item) }
        return true
    }
}



// MARK: - 删除
extension DetectRecordBloodPressure {
    ///: - 移除表
    internal static func remove() {
        guard let _dbQueue = dbQueue else { return }
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try db.drop(table: databaseTableName)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    /// 删除所有
    internal static func deleteAll() {
        _latestRecord = nil
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try DetectRecordBloodPressure.deleteAll(db)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
}
