//
//  BluetoothCustomer.swift
//  cttoll
//
//  Created by Mac on 2020/3/30.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import CoreBluetooth
import RxBluetoothKit

class BleCustomer: NSObject {
    
    // MARK: - UUID
    
    static let humanServiceUUID     = CBUUID(string: "1812");
    static let sw294ServiceUUID   = CBUUID(string: "180D");
    static let batteryServiceUUID   = CBUUID(string: "180F");
    static let heartRateServiceUUID = CBUUID(string: "1812");

    static let serviceUUID               = CBUUID(string: "6E400001-B5A3-F393-E0A9-E50E24DCCA9E");
    static let writeCharacteristicUUID   = CBUUID(string: "6E400002-B5A3-F393-E0A9-E50E24DCCA9E");
    static let readCharacteristicUUID    = CBUUID(string: "6E400003-B5A3-F393-E0A9-E50E24DCCA9E");

    static let serviceUUIDdfu   = CBUUID(string: "FE59"); // Dfu 服务 UUID
    static let serviceUUIDSX    = CBUUID(string: "F618"); // S 系列服务 UUID
    
    // MARK: - convenience UUID

    static let scanUUIDs = [sw294ServiceUUID, heartRateServiceUUID, humanServiceUUID, serviceUUIDdfu] ;
    static let retrieveConnectedUUIDs = [serviceUUID, serviceUUIDdfu];
    static let retrieveConnectedUUIDs_s = [serviceUUIDSX];
    static let serviceUUIDs = [serviceUUID];
    static let characteristicUUIDs = [writeCharacteristicUUID, readCharacteristicUUID];
    
    static let dfuScanUUIDs = [serviceUUIDdfu] ;
    static let dfuRetrieveConnectedUUIDs = [serviceUUIDdfu];
    static let dfuServiceUUIDs = [serviceUUIDdfu];

    
    enum DeviceService: String, ServiceIdentifier {
        case info =  "6E400001-B5A3-F393-E0A9-E50E24DCCA9E"
        var uuid: CBUUID {
            return CBUUID(string: self.rawValue)
        }
    }
    
    enum DeviceCharacteristic: String, CharacteristicIdentifier {
        case writeInfo = "6E400002-B5A3-F393-E0A9-E50E24DCCA9E"
        case readInfo  = "6E400003-B5A3-F393-E0A9-E50E24DCCA9E"
        var uuid: CBUUID {
            return CBUUID(string: self.rawValue)
        }
        var service: ServiceIdentifier {
            return DeviceService.info
        }
    }
    
    
}

// MARK: - CentralManager 初始化 convenience

extension BleCustomer {
    
    /// CentralManager（蓝牙中心端管理） 初始化
    /// - Parameters:
    ///   - powerAlert: 未开启蓝牙，是否弹出系统提示框
    ///   - onWillRestoreCentralManagerState: 已恢复状态时调用闭包.
    static func initCentralManager(show powerAlert: Bool = true,
                                   onWillRestoreCentralManagerState: OnWillRestoreCentralManagerState? = nil) -> CentralManager {
        
        let info = Bundle.main.infoDictionary
        let bundleId = info?["CFBundleIdentifier"] as? String
        let lab = "\(bundleId ?? "com.bandkirpro").centralmanager";
        let queue = DispatchQueue(label: lab)
        var options: [String: Any] = [CBCentralManagerOptionShowPowerAlertKey: powerAlert]

        /// 支持 UIBackgroundMode： bluetooth-central
        let bgmodeItems = info?["UIBackgroundModes"] as? [String]
        let supportBackgroundMode: Bool = bgmodeItems?.contains("bluetooth-central") ?? false
        if supportBackgroundMode {
            if onWillRestoreCentralManagerState != nil {
                let restoreIdentifier =  "\(bundleId ?? "com.bandkirpro").restoreIdentifier";
                options[CBCentralManagerOptionRestoreIdentifierKey] = restoreIdentifier
            }
        }
        return CentralManager(queue:queue, options: options, onWillRestoreCentralManagerState: onWillRestoreCentralManagerState)
    }
    
    /// CentralManager（蓝牙中心端管理） 初始化，升级模式使用
    /// - Parameters:
    ///   - powerAlert: 未开启蓝牙，是否弹出系统提示框
    ///   - onWillRestoreCentralManagerState: 已恢复状态时调用闭包.
    static func initDfuCentralManager(show powerAlert: Bool = false, onWillRestoreCentralManagerState: OnWillRestoreCentralManagerState? = nil) -> CentralManager {
        let info = Bundle.main.infoDictionary
        let bundleId = info?["CFBundleIdentifier"] as? String
        let lab = "\(bundleId ?? "com.bandkirpro").centralmanager.dfu";
        let queue = DispatchQueue(label: lab)
        var options: [String: Any] = [CBCentralManagerOptionShowPowerAlertKey: powerAlert]

        /// 支持 UIBackgroundMode： bluetooth-central
        let bgmodeItems = info?["UIBackgroundModes"] as? [String]
        let supportBackgroundMode: Bool = bgmodeItems?.contains("bluetooth-central") ?? false
        if supportBackgroundMode {
            if onWillRestoreCentralManagerState != nil {
                let restoreIdentifier =  "\(bundleId ?? "com.bandkirpro").restoreIdentifier";
                options[CBCentralManagerOptionRestoreIdentifierKey] = restoreIdentifier
            }
        }
        return CentralManager(queue:queue, options: options, onWillRestoreCentralManagerState: onWillRestoreCentralManagerState)
    }
}

