//
//  SRDfuService.swift
//  CTFit
//
//  Created by Mac on 2020/8/5.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation
import CoreBluetooth

import RxSwift
import RxBluetoothKit
import Repeater
import iOSDFULibrary

public class SRDfuService {
    
    public enum SearchStatus {
        case searching, discovered, notFound
    }
    
    public typealias SearchMatchPeripheralClosure = (_ peripheral: SRPeripheral) -> Bool
    
    public typealias SearchStatusClosure = (_ status: SearchStatus) -> Void
    
    public typealias UploadProgressClosure = (_ progress: Int, _ uploadedBytes: Int, _ totalBytes: Int) -> Void

    public typealias DfuStatusClosure = (_ state: DFUState) -> Void

    public typealias DfuErrorMessageClosure = (_ error: DFUError, _ message: String) -> Void

    public static let shared = SRDfuService()

    
    // MARK: - Private fields

    private let disposeBag = DisposeBag()

    private let scheduler: ConcurrentDispatchQueueScheduler

    private var scanningDisposable: Disposable? = nil

    private let scanningSubject = PublishSubject<SRResult<SRPeripheral, Error>>()

    private var scanningOutput: Observable<SRResult<SRPeripheral, Error>> {
        return scanningSubject.share(replay: 1, scope: .forever).asObservable()
    }
    
    
        
    private let centralManager:CentralManager = BleCustomer.initDfuCentralManager()

    private var selectPeripheral: Peripheral?
    
    private var dfuService: DFUServiceController?
    
    private var searchMatchPeripheralClosure: SearchMatchPeripheralClosure?

    private var searchStatusClosure: SearchStatusClosure?
    
    private var progressClosure: UploadProgressClosure?

    private var dfuStatusClosure: DfuStatusClosure?

    private var dfuErrorMessageClosure: DfuErrorMessageClosure?
    
    // MARK: - 订阅
    /// 蓝牙状态
    private lazy var onceCentralState: Void = {
        centralManager.observeState()
            .subscribeOn(MainScheduler.asyncInstance)
            .subscribe(onNext: { [weak self] (state) in
                guard let strong = self else { return }
            }).disposed(by: disposeBag)
    }()
    
    // MARK: - Initialize
    private func Initialize() {
        onSearching()
    }
    
    private init() {
        let timerQueue = DispatchQueue(label: "com.jpaxh.ctbandkit.centralManager.dfu.timer")
        scheduler = ConcurrentDispatchQueueScheduler(queue: timerQueue)
        Initialize()
    }

    private func onSearching() {
        scanningOutput.observeOn(MainScheduler.instance).subscribe(
            onNext: {[weak self] result in
                guard let strong = self else { return }
                guard let closure = strong.searchMatchPeripheralClosure else { return }
                guard let statusClosure = strong.searchStatusClosure else { return }
                switch result {
                case .success(let value):
                    guard closure(value) else { return }
                    BandLog.i("成功: \(value)")
                    strong.selectPeripheral = value.peripheral
                    DispatchQueue.main.async {
                        statusClosure(.discovered)
                        strong.stopScanning()
                    }

                case .error(let error):
                    BandLog.e("失败: \(error)")
                    DispatchQueue.main.async {
                        statusClosure(.notFound)
                    }
                }
            }
        ).disposed(by: disposeBag)
    }
}

// MARK: - Private method

extension SRDfuService {
    
    /// 扫描周边外围DFU设备
    private func startScan() {
        scanningDisposable = centralManager.observeState()
        .startWith(centralManager.state)
        .subscribeOn(MainScheduler.instance)
        .timeout(.seconds(15), scheduler: scheduler) //
        .flatMap { [weak self] state -> Observable<ScannedPeripheral> in
            guard let strong = self else { return Observable.empty() }
            // 如果蓝牙state != .powerdOn 者直接发出错误
            if state != .poweredOn, let error = BluetoothError(state: state) {
                let subject = PublishSubject<ScannedPeripheral>()
                subject.onError(error)
                return subject
            }
            // 合并
            return Observable.of(strong.retrieveObservable(servicesUUIDs: BleCustomer.dfuRetrieveConnectedUUIDs),
                                 strong.centralManager.scanForPeripherals(withServices: BleCustomer.dfuScanUUIDs)).merge()
            
        }
        .subscribe(onNext: { [weak self] scannedPeripheral in
                self?.scanningSubject.onNext(.success(SRPeripheral(peripheral: scannedPeripheral)))
            }, onError: { [weak self] error in
                self?.scanningSubject.onNext(.error(error))
            }
        )
    }
    
    /// 扫描周边外围DFU设备
    private func startScan(peripheral identifier: UUID) {
        scanningDisposable = centralManager.observeState()
        .startWith(centralManager.state)
        .subscribeOn(MainScheduler.asyncInstance)
        .timeout(.seconds(5), scheduler: scheduler) //
        .flatMap { [weak self] state -> Observable<ScannedPeripheral> in
            guard let strong = self else { return Observable.empty() }
            // 如果蓝牙state != .powerdOn 者直接发出错误
            if state != .poweredOn, let error = BluetoothError(state: state) {
                let subject = PublishSubject<ScannedPeripheral>()
                subject.onError(error)
                return subject
            }
            // 合并
            return Observable.of(strong.retrieveObservable(withIdentifier: identifier),
                                 strong.centralManager.scanForPeripherals(withServices: [CBUUID(nsuuid: identifier)])).merge()
            
        }
        .subscribe(onNext: { [weak self] scannedPeripheral in
            guard let strong = self else { return }
            strong.scanningSubject.onNext(.success(SRPeripheral(peripheral: scannedPeripheral)))
        }, onError: { [weak self] error in
            guard let strong = self else { return }
            strong.scanningSubject.onNext(.error(error))
        })
    }
    
    
    /// 如果您希望停止对外围设备的扫描，您需要释放scanningDisposable对象
    /// 当您订阅由centralManager.scanForPeripherals(:_)返回的可观察对象的事件时创建，或者您将一个观察者绑定到它。
    /// 请查看上面的starscan()以获得详细信息
    private func stopScan() {
        if let hasScanningDisposable = scanningDisposable, centralManager.manager.isScanning {
            self.scanningSubject.onNext(.error(BandServiceError.scanningStop))
            hasScanningDisposable.dispose()
        }
    }
    
    /// 将已连接到 CentralManager 的设备列表 计入 扫描设备列表内
    /// - Parameter servicesUUIDs: 指定的服务 UUIDs
    private func retrieveObservable(servicesUUIDs: [CBUUID]) -> Observable<ScannedPeripheral>{
        let retrieveConnectedPeripherals = self.centralManager.retrieveConnectedPeripherals(withServices: servicesUUIDs)
        return PublishSubject.from(retrieveConnectedPeripherals).map { (peripheral) -> ScannedPeripheral in
                ScannedPeripheral(peripheral: peripheral)
        }
    }
    
    private func retrieveObservable(withIdentifier identifier: UUID) -> Observable<ScannedPeripheral>{
        let retrieveConnectedPeripherals = self.centralManager.retrieveConnectedPeripherals(withServices: BleCustomer.dfuRetrieveConnectedUUIDs)
        let items = retrieveConnectedPeripherals.filter { (peripheral) -> Bool in
            (peripheral.identifier == identifier)
        }
        return PublishSubject.from(items).map { (peripheral) -> ScannedPeripheral in
                ScannedPeripheral(peripheral: peripheral)
        }
    }
}

// MARK: - Public method

extension SRDfuService {
    
    /// - 切换到 DUF 模式
    public func switchToDfuMode(_ callback: @escaping SRWriteClosure) {
        CmdRequest.clear()
        DispatchQueue.main.asyncAfter(deadline: .now() + 1) {
            SRCommand.firmwareControl(type: .upgrade).writeToDevice()
        }
        DispatchQueue.main.asyncAfter(deadline: .now() + 3) {
            SRCommand.firmwareControl(type: .upgrade).writeToDevice()
        }
    }
    
    /// - 扫描外围设备
    public func startScanning(peripheralClosure: @escaping SearchMatchPeripheralClosure, statusClosure: @escaping SearchStatusClosure) {
        selectPeripheral = nil
        stopScanning()
        searchMatchPeripheralClosure = peripheralClosure
        searchStatusClosure = statusClosure
        DispatchQueue.main.asyncAfter(deadline: .now()+1.5) { [unowned self] in
            statusClosure(.searching)
            self.startScan()
        }
    }
    
    /// - 结束外设扫描
    public func stopScanning() {
        searchMatchPeripheralClosure = nil
        searchStatusClosure = nil
        stopScan()
    }
    
    /// - 开始上传固件升级
    public func startUpLoadFMFile(filePath: URL, peripheral: SRPeripheral, progressClosure: @escaping UploadProgressClosure, dfuStatusClosure: @escaping DfuStatusClosure, dfuErrorMessageClosure: @escaping DfuErrorMessageClosure) {
        guard let selectedFirmware = DFUFirmware(urlToZipFile:filePath) else {
            BandLog.v("Wrong file: \(filePath)")
            dfuErrorMessageClosure(.fileInvalid, "fileInvalid")
            return
        }
        self.progressClosure = progressClosure
        self.dfuStatusClosure = dfuStatusClosure
        self.dfuErrorMessageClosure = dfuErrorMessageClosure
        
        let initiator = DFUServiceInitiator().with(firmware: selectedFirmware)
        initiator.logger = self // - to get log info
        initiator.delegate = self // - to be informed about current state and errors
        initiator.progressDelegate = self // - to show progress bar
        dfuService = initiator.start(target: peripheral.peripheral.peripheral)
    }
    
    /// - 取消固件上传
    public func cancelUpLoadFMFile() {
        guard let hasService = dfuService, !hasService.aborted else { return }
        _ = hasService.abort()
    }
    
    public func retrievePeripherals(withIdentifier identifier: UUID) -> SRPeripheral? {
        let retrieveConnectedPeripherals = self.centralManager.retrieveConnectedPeripherals(withServices: BleCustomer.dfuRetrieveConnectedUUIDs)
        for peripheral in retrieveConnectedPeripherals {
            if peripheral.identifier == identifier {
                return SRPeripheral(peripheral: peripheral)
            }
        }
        let retrievePeripherals = self.centralManager.retrievePeripherals(withIdentifiers: [identifier])
        for peripheral in retrievePeripherals {
            if peripheral.identifier == identifier {
                return SRPeripheral(peripheral: peripheral)
            }
        }
        return nil
    }
}

// MARK: - Delegate

extension SRDfuService: LoggerDelegate, DFUServiceDelegate, DFUProgressDelegate {
    
    public func logWith(_ level: LogLevel, message: String) {
        //BandLog.v("\(level.name()): \(message)")
    }
    
    public func dfuStateDidChange(to state: DFUState) {
        BandLog.v("\(state.description())")
        guard let closure = dfuStatusClosure else { return }
        DispatchQueue.main.async { closure(state) }
    }
    
    public func dfuError(_ error: DFUError, didOccurWithMessage message: String) {
        BandLog.e("\(error): \(message)")
        guard let closure = dfuErrorMessageClosure else { return }
        DispatchQueue.main.async { closure(error, message) }
    }
    
    public func dfuProgressDidChange(for part: Int, outOf totalParts: Int, to progress: Int, currentSpeedBytesPerSecond: Double, avgSpeedBytesPerSecond: Double) {
        BandLog.v("Firmware file upload progress. \(progress)%")
        guard let closure = progressClosure else { return }
        DispatchQueue.main.async { closure(progress, part, totalParts) }
    }
}

