import Foundation
import CoreBluetooth

import RxSwift
import RxBluetoothKit
import Repeater
import NEUtils

public enum SRResult<T, E> {
    case success(T)
    case error(E)
}

/// SRKitService是一个封装逻辑的类，用于在 CentralManager 对象上执行的大多数操作。
/// 在这里，您可以看到诸如扫描外设、发现服务和发现外设等功能的示例用法。
final public class SRKitService {
    
    public typealias  Disconnection = (Peripheral, DisconnectionReason?)
    public typealias  EstablishConnectionClosure =  (SRResult<SRPeripheral, Error>) -> (Void)

    
    private init() {
        let timerQueue = DispatchQueue(label: "com.jpaxh.bandkit.centralManager.timer")
        scheduler = ConcurrentDispatchQueueScheduler(queue: timerQueue)
        Initialize();
    }
    
    private func Initialize() {
        subscribeCentralState()
        subscribeScanning()
        subscribeEstablishConnection()
        subscribeDisconnect()
        subscribeValueAndNotification()
        subscribeApplicationNotify()
        DispatchQueue.main.async { _ = SRDataManager.shared }
        DispatchQueue.main.async { _ = SRDfuService.shared }
    }
    

    // MARK: - Private fields
    
    private let disposeBag = DisposeBag()
    
    private let scheduler: ConcurrentDispatchQueueScheduler

    private let centralManager:CentralManager = BleCustomer.initCentralManager()
    
    private var centralState: BluetoothState { centralManager.state }
    
    private var autoConnecting: Bool = true
    
    private var bindPeripheral: SRBindPeripheral? { SRBindPeripheral.hasBinded }
    
    private var isConnected: Bool {  return activePeripheral?.isConnected ?? false }
    
    private var connectingPeripheral: Peripheral?

    private var activePeripheral: Peripheral?
    
    private var currentPeripheral: SRPeripheral? {
        guard let peripheral =  activePeripheral else { return nil}
        return SRPeripheral(peripheral: peripheral)
    }

    
    private var readCharacteristic:Characteristic? {
        characteristic(with: BleCustomer.DeviceCharacteristic.readInfo, in: activePeripheral);
    }
    
    private var writeCharacteristic:Characteristic? {
        characteristic(with: BleCustomer.DeviceCharacteristic.writeInfo, in: activePeripheral);
    }
    
    private lazy var disconnectReconnectTimer: Repeater = {
        let repeater = Repeater.once(after: .seconds(1), queue: DispatchQueue.main) { [unowned self] (_) in
            self.reconnectBindPeripheral()
        }
        repeater.pause()
        return repeater
    }()
    

    // MARK: - Private subjects
    
    private let centralStateSubject = PublishSubject<BluetoothState>()

    private let scanningSubject = PublishSubject<SRResult<SRPeripheral, Error>>()
    
    private let establishConnectionSubject = PublishSubject<SRResult<SRPeripheral, Error>>()
        
    private let disconnectionSubject = PublishSubject<(Peripheral, DisconnectionReason?)>()
    
    private let connectionEstablishSubject = PublishSubject<SRConnectEstablishState>()
    
    private let connectionStateSubject = PublishSubject<SRConnectState>()

    
    private let readValueSubject = PublishSubject<SRResult<Characteristic, Error>>()

    private let writeValueSubject = PublishSubject<SRResult<Characteristic, Error>>()

    private let isNotifyingSubject =  PublishSubject<Bool>()
    
    private let updatedValueAndNotificationSubject = PublishSubject<Characteristic>()
    
    private let updatedNameSubject = PublishSubject<(Peripheral, String?)>()
    


    // MARK: - Private disposable

    private var scanningDisposable: Disposable?

    private var connectionDisposables: [Peripheral: Disposable] = [:]
    
    private var disconnectinDisposables: [Peripheral: Disposable] = [:]

    private var notificationDisposables: [Characteristic: Disposable] = [:]
    
    private var isNotifyDisposables: [Characteristic: Disposable] = [:]
    
    private var updateNameDisposables: [Peripheral: Disposable] = [:]
    
}
    

// MARK: - 公有的类成员
extension SRKitService {
    
    static let shared = SRKitService()
    public static var isConnected: Bool {  shared.isConnected }
    public static var centralState: BluetoothState { shared.centralState }
    public static var currentPeripheral: SRPeripheral? { shared.currentPeripheral }
    public static var autoConnecting: Bool {
        get { shared.autoConnecting }
        set { shared.autoConnecting = newValue }
    }
    
    
    ///: - Output
    public static var bluetoothStateOutput: PublishSubject<BluetoothState> {
        shared.centralStateSubject }
    public static var scanningOutput: Observable<SRResult<SRPeripheral, Error>> {
        shared.scanningSubject.share(replay: 1, scope: .forever).asObservable().observeOn(MainScheduler.asyncInstance) }
        
    public static var connectionEstablishOutput: Observable<SRConnectEstablishState> {
        shared.connectionEstablishSubject.asObservable() }
    public static var connectionStateOutput: Observable<SRConnectState> {
        shared.connectionStateSubject.asObservable() }
    
    public static var isNotifyingOutput:  Observable<Bool> {
        shared.isNotifyingSubject.asObservable() }
    public static var updatedValueAndNotificationOutput: Observable<Characteristic> {
        shared.updatedValueAndNotificationSubject.asObservable() }
    
    public static var updatedNameOutput: Observable<(Peripheral, String?)> {
        shared.updatedNameSubject.asObservable() }
    
    
    /// : - 绑定设备（用于自动重连）
    public static var  binded: Bool { (bindPeripheral != nil) }
    public static var  bindPeripheral: SRBindPeripheral? { shared.bindPeripheral }
    public static func bind() { shared.bind() }
    public static func unbind() { shared.unbind() }

    /// : - 写数据包
    public static func writeValue(with data: Data) -> Bool { shared.writeValue(with: data) }
    
    /// : - 查找设备
    public static func startScanning(_ time: Int) { shared.startScanning(time) }
    public static func stopScanning() { shared.stopScanning() }
    public static func retrieveConnectedPeripherals() -> [SRPeripheral] { shared.retrieveConnectedPeripherals() }
    public static func retrieveConnectedPeripherals_s() -> [Peripheral] { shared.retrieveConnectedPeripherals_s() }
    public static func retrievePeripherals(withIdentifier identifier: UUID) -> SRPeripheral? {
        shared.retrievePeripherals(withIdentifier: identifier) }
    
    
    /// : - 连接设备
    public static func disconnect() { shared.disconnect() }
    public static func reconnectBindPeripheral() { shared.reconnectBindPeripheral() }
    public static func connect(for peripheral: SRPeripheral, time: TimeInterval = 0, output:EstablishConnectionClosure? = nil) {
        shared.establishConnection(for: peripheral, time: time, output: output)}

}


// MARK: - 订阅

extension SRKitService {

    /// 蓝牙状态
    private func subscribeCentralState() {
        centralManager.observeState()
            .subscribeOn(MainScheduler.instance)
            .subscribe(onNext: { [weak self] (state) in
                guard let strong = self else { return }
                BandLog.i("centralManager(state: \(state))")
                strong.reconnectBindPeripheral()
                strong.centralStateSubject.onNext(state)
            }).disposed(by: disposeBag)
    }
    
    /// 扫描设备
    private func subscribeScanning() {
        scanningSubject.asObservable()
            .observeOn(MainScheduler.instance)
            .subscribe(onNext: { result in
                switch result {
                case .success(_): break
                case .error(let error): BandLog.i("Scanning(end: \(error))")
                }
            }).disposed(by: disposeBag)
    }
    
    /// 建立连接订阅
    private func subscribeEstablishConnection() {
        establishConnectionSubject.asObservable().subscribe(onNext: {[weak self] result in
            guard let strong = self else { return }
            if case .success(let searchPeripheral) = result {
                let peripheral = searchPeripheral.peripheral
                
                let maxLenResponse = peripheral.maximumWriteValueLength(for: .withResponse)
                let maxLenResponseWithout = peripheral.maximumWriteValueLength(for: .withoutResponse)
                CmdHolder.maxLenResponse = UInt16(truncatingIfNeeded: maxLenResponse)
                CmdHolder.maxLenResponseWithout = UInt16(truncatingIfNeeded: maxLenResponseWithout)
                BandLog.v(("""
                establishConnection(success: \(searchPeripheral.peripheralName)）
                maximumWriteValueLength.withResponse: \(maxLenResponse)
                maximumWriteValueLength.withoutResponse: \(maxLenResponseWithout)
                """))
                
                strong.activePeripheral = peripheral;
                DispatchQueue.main.async {
                    strong.observeDisconnect(for: peripheral)
                    strong.observeUpdateName(for: peripheral)
                    if let characteristic = strong.readCharacteristic {
                        strong.observeIsNotify(peripheral: searchPeripheral.peripheral, characteristic: characteristic)
                        strong.observeValueUpdateAndSetNotification(for: characteristic)
                    }
                    
                    strong.connectionEstablishSubject.onNext(.successed)
                    strong.connectionStateSubject.onNext(.connected)
                }
            }
                
            if case .error(let error) = result {
                switch error {
                case BluetoothError.peripheralConnectionFailed(let peripheral, let error):
                    BandLog.e("""
                        establishConnection(failed: \(String(describing: peripheral.name)),
                        error: \(String(describing: error)))
                        """)
                    if let hasError = error {
                        // hasError._code == 14 配对丢失
                        strong.connectionEstablishSubject.onNext(.failed(.code(hasError._code)))
                    } else {
                        strong.connectionEstablishSubject.onNext(.failed(.code(0)))
                    }
                case BandServiceError.establishConnectionTimeout: strong.connectionEstablishSubject.onNext(.failed(.timeout))
                default: strong.connectionEstablishSubject.onNext(.failed(.bleoff))
                }
                
                if let peripheral = strong.connectingPeripheral {
                    strong.disconnect(peripheral)
                    strong.disconnectReconnectTimer.reStart()
                }
            }
            
        }).disposed(by: disposeBag)
    }
    
        
    /// 设备断开连接订阅
    private func subscribeDisconnect() {
        disconnectionSubject.asObservable().subscribe(onNext: { [weak self] (peripheral, reason) in
            guard let strong = self else { return }
            BandLog.i("""
                Disconnected: \(String(describing: peripheral.name))
                Reason: \(String(describing: reason))
                """)
            DispatchQueue.main.async {
                strong.disposeDisconnect(for: peripheral)
                strong.disposeUpdateName(for: peripheral)
                if let characteristic = strong.readCharacteristic {
                    strong.disposeIsNotify(for: characteristic)
                }
            }
            strong.connectionStateSubject.onNext(.disconnected(reason?.localizedDescription ?? ""))
            strong.disconnectReconnectTimer.reStart()
        }).disposed(by: disposeBag)
    }
    
    /// 收到数据订阅
    private func subscribeValueAndNotification() {
        updatedValueAndNotificationSubject.asObservable().subscribe(onNext: {[weak self] characteristic in
            guard let strong = self else { return }
            strong.received(characteristic)
        }).disposed(by: disposeBag)
    }
        
    /// 应用生命循环通知
    private func subscribeApplicationNotify() {
        NEAppNotifyUtils.observable.subscribe(onNext: { [weak self] type in
            guard let strong = self else { return }
            switch type {
            case .willEnterForeground: strong.reconnectBindPeripheral()
            default: break
            }
        }).disposed(by: disposeBag)
    }
}

// MARK: - 扫描外围设备

extension SRKitService {

    /// 扫描周边外围设备
    /// 从观察 CentralManager.state 开始,  并用 startwith(:_) 操作符加入初始状态 CentralManager.state
    /// 将已连接到 CentralManager 的设备列表 计入 扫描设备列表内 ( 业务需求 )
    private func startScanning(_ time: Int) {
        scanningDisposable = centralManager.observeState()
            .startWith(centralManager.state)
            .subscribeOn(MainScheduler.asyncInstance)
            .timeout(.seconds(time), scheduler: scheduler)
            .flatMap { [weak self] state -> Observable<ScannedPeripheral> in
                guard let strong = self else { return Observable.empty() }
                // 如果蓝牙state != .powerdOn 直接发出错误
                if state != .poweredOn, let error = BluetoothError(state: state) {
                    let subject = PublishSubject<ScannedPeripheral>()
                    subject.onError(error)
                    return subject
                }
                // 合并
                return Observable.of(strong.retrieveObservable(servicesUUIDs: BleCustomer.retrieveConnectedUUIDs),
                                     strong.centralManager.scanForPeripherals(withServices: BleCustomer.scanUUIDs)).merge()
            }
            .subscribe(onNext: { [weak self] scannedPeripheral in
                guard let strong = self else { return }
                strong.scanningSubject.onNext(.success(SRPeripheral(peripheral: scannedPeripheral)))
            }, onError: { [weak self] error in
                guard let strong = self else { return }
                strong.scanningSubject.onNext(.error(error))
            })
    }
    
    /// 如果您希望停止对外围设备的扫描，您需要释放scanningDisposable对象
    /// 当您订阅由centralManager.scanForPeripherals(:_)返回的可观察对象的事件时创建，或者您将一个观察者绑定到它。
    /// 请查看上面的starscan()以获得详细信息
    private func stopScanning() {
        if let hasScanningDisposable = scanningDisposable, centralManager.manager.isScanning {
            self.scanningSubject.onNext(SRResult.error(BandServiceError.scanningStop))
            hasScanningDisposable.dispose()
            scanningDisposable = nil
        }
    }
        
    private func retrieveConnectedPeripherals() -> [SRPeripheral] {
        let retrieveConnectedPeripherals = self.centralManager.retrieveConnectedPeripherals(withServices: BleCustomer.retrieveConnectedUUIDs)
        return retrieveConnectedPeripherals.map { SRPeripheral(peripheral: $0) }
    }
    
    private func retrieveConnectedPeripherals_s() -> [Peripheral] {
        let retrieveConnectedPeripherals = self.centralManager.retrieveConnectedPeripherals(withServices: BleCustomer.retrieveConnectedUUIDs_s)
        return retrieveConnectedPeripherals
    }
    
    private func retrievePeripherals(withIdentifier identifier: UUID) -> SRPeripheral? {
        let retrieveConnectedPeripherals = self.centralManager.retrieveConnectedPeripherals(withServices: BleCustomer.retrieveConnectedUUIDs)
        for peripheral in retrieveConnectedPeripherals {
            if peripheral.identifier == identifier {
                return SRPeripheral(peripheral: peripheral)
            }
        }
        let retrievePeripherals = self.centralManager.retrievePeripherals(withIdentifiers: [identifier])
        for peripheral in retrievePeripherals {
            if peripheral.identifier == identifier {
                return SRPeripheral(peripheral: peripheral)
            }
        }
        return nil
    }
}

// MARK: - 外围设备连接建立

extension SRKitService {

    /// 外围设备连接后发现所需服务以及特征，可设置超时处理
    /// 当您发现一项服务时，首先需要与外围设备建立连接。 然后为该外围对象调用discoverServices(:_)，
    private func establishConnection(for device: SRPeripheral, time: TimeInterval = 0, output:EstablishConnectionClosure? = nil) {
        BandLog.i("To Connection: \(device.peripheralName)")
        let peripheral = device.peripheral;
        let isConnected = peripheral.isConnected
        self.connectingPeripheral = peripheral

        var connectedObservableCreator = {
            peripheral.service(with: BleCustomer.DeviceService.info).asObservable().flatMap {
                $0.discoverCharacteristics(BleCustomer.characteristicUUIDs)
            }
        }
        
        var connectObservableCreator = {
            peripheral.establishConnection()
                .flatMap {
                    $0.service(with: BleCustomer.DeviceService.info)
                }
                .flatMap {
                    $0.discoverCharacteristics(BleCustomer.characteristicUUIDs)
                }
        }
        
        if time > 0.1 {
            connectedObservableCreator = {
                peripheral.service(with: BleCustomer.DeviceService.info).timeout(.seconds(Int(time)), scheduler: self.scheduler).asObservable().flatMap {
                    $0.discoverCharacteristics(BleCustomer.characteristicUUIDs).timeout(.seconds(Int(time)), scheduler: self.scheduler)
                }
            }
            connectObservableCreator = {
                peripheral.establishConnection()
                    .flatMap {
                        $0.service(with: BleCustomer.DeviceService.info).timeout(.seconds(Int(time)), scheduler: self.scheduler)
                    }
                    .flatMap {
                        $0.discoverCharacteristics(BleCustomer.characteristicUUIDs).timeout(.seconds(Int(time)), scheduler: self.scheduler)
                    }
            }
        }
        
        let timer = Repeater.once(after: .seconds(time)) { (_) in
            if time > 0.1 {
                self.disconnect(peripheral)
                self.establishConnectionSubject.onNext(.error(BandServiceError.establishConnectionTimeout))
                if let block = output { block(.error(BandServiceError.establishConnectionTimeout)) }
            }
        }
        
        let observable = isConnected ? connectedObservableCreator(): connectObservableCreator()
        let disposable = observable.subscribe(
            onNext: {  [weak self] characteristics in
                if time > 0.1 { timer.pause(); }
                guard let strong = self else { return }
                strong.establishConnectionSubject.onNext(.success(device))
                if let block = output { block(.success(device)) }
            },
            onError: { [weak self] error in
                if time > 0.1 { timer.pause(); }
                guard let strong = self else { return }
                strong.establishConnectionSubject.onNext(.error(error))
                if let block = output { block(.error(error)) }
            }
        )
       
        connectionDisposables[peripheral] = disposable
    }

    /// 重连绑定设备
    private func reconnectBindPeripheral() {
        guard autoConnecting else { return }
        guard centralManager.state == .poweredOn else { return }
        
        let time:TimeInterval = 10.0
        if let peripheral = activePeripheral  {
            guard !peripheral.isConnected else { return }
            let searchPeripheral = SRPeripheral(peripheral: peripheral)
            establishConnection(for: searchPeripheral, time: time)
            return
        }
        
        guard let hasBindPeripheral = bindPeripheral, let identifier = UUID(uuidString: hasBindPeripheral.uuid),
            let peripheral = retrievePeripherals(withIdentifier: identifier) else { return }
        establishConnection(for: peripheral, time: time)
    }
    
    /// 断开当前连接
    private func disconnect() {
        guard let peripheral = activePeripheral else { return }
        disconnect(peripheral)
    }
}

// MARK: - Method for operation

extension SRKitService {
    
    /// 绑定设备
    private func bind() {
        guard let peripheral = currentPeripheral else { return }
        let bindPeripheral = SRBindPeripheral(with: peripheral.uuid, name: peripheral.peripheralName)
        bindPeripheral.save()
    }
    
    /// 解除绑定
    private func unbind() {
        SRDataManager.unbindClear()
        if let hasBindPeripheral = bindPeripheral { hasBindPeripheral.remove()}
        disconnect(); activePeripheral = nil
    }

    /// 写数据
    @discardableResult
    private func writeValue(with data: Data) -> Bool {
        guard let characteristic =  writeCharacteristic else { return false; }
        self.writeValueTo(characteristic: characteristic, data: data)
        return true;
    }
    
    /// 接收数据
    private func received(_ characteristic: Characteristic) {
        guard self.readCharacteristic == characteristic,
              let value = characteristic.value, value.count > 0  else { return }
        do {
            try CmdRequest.received(value)
        } catch {
            BandLog.e("""
                received:(value: \(String(describing: characteristic.value?.logDescription)),
                error: \(String(describing: error))
                """)
        }
    }
}

extension SRKitService {

    /// 处理给定的连接，即从外围设备自动断开连接，
    /// 因此，首先要从perpiheral上断开连接，
    /// 然后从外围设备的集合中删除已断开的外围设备.
    private func disconnect(_ peripheral: Peripheral) {
        guard let disposable = connectionDisposables[peripheral] else { return }
        disposable.dispose()
        connectionDisposables[peripheral] = nil
    }
    
    /// 将已连接到 CentralManager 的设备列表
    /// - Parameter servicesUUIDs: 指定的服务 UUIDs
    private func retrieveObservable(servicesUUIDs: [CBUUID]) -> Observable<ScannedPeripheral>{
        let retrieveConnectedPeripherals = self.centralManager.retrieveConnectedPeripherals(withServices: servicesUUIDs)
        return PublishSubject.from(retrieveConnectedPeripherals).map { (peripheral) -> ScannedPeripheral in
                ScannedPeripheral(peripheral: peripheral)
        }
    }

    /// 查找 Peripheral 中指定 Characteristic
    private func characteristic(with identifier: CharacteristicIdentifier, in selPeripheral: Peripheral?) -> Characteristic?{
        if  let peripheral = selPeripheral,
            let services = peripheral.services,
            let service  = services.first(where: { $0.uuid == identifier.service.uuid }),
            let characteristics = service.characteristics,
            let characteristic  = characteristics.first(where: { $0.uuid == identifier.uuid }){
            return characteristic
        }
        return nil;
    }
    
    /// 读取Characteristic
    private func readValueFrom(_ characteristic: Characteristic) {
        characteristic.readValue().subscribe(onSuccess: { [unowned self] characteristic in
            self.readValueSubject.onNext(.success(characteristic))
        }, onError: { [unowned self] error in
            self.readValueSubject.onNext(.error(error))
        }).disposed(by: disposeBag)
    }

    /// 写入Characteristic
    private func writeValueTo(characteristic: Characteristic, data: Data) {
        guard let writeType = characteristic.determineWriteType() else {
            BandLog.e("characteristic not writeType")
            return
        }
        characteristic.writeValue(data, type: writeType).subscribe(onSuccess: { [unowned self] characteristic in
            self.writeValueSubject.onNext(.success(characteristic))
        }, onError: { [unowned self] error in
            self.writeValueSubject.onNext(.error(error))
        }).disposed(by: disposeBag)
    }
}

// MARK: - Private: Observable/Disposables

extension SRKitService {

    /// observeNotifyValue 告诉我们一个特征什么时候改变了它的状态(e.g isNotifying)。
    /// 我们需要使用这种方法，因为硬件需要大量的时间来切换特性的状态.
    private func observeIsNotify(peripheral: Peripheral, characteristic: Characteristic) {
        if let _ = isNotifyDisposables[characteristic] { return }
        let disposable = peripheral.observeNotifyValue(for: characteristic).subscribe(
            onNext: { [weak self] (characteristic) in
                guard let strong = self else { return}
                BandLog.i("\(peripheral.peripheral): \(characteristic.uuid) isNotifying: \(characteristic.isNotifying)")
                strong.isNotifyingSubject.onNext(characteristic.isNotifying)
            },
            onError: { _ in }
        )
        isNotifyDisposables[characteristic] = disposable
    }
    
    private func disposeIsNotify(for characteristic: Characteristic) {
        if let disposable = isNotifyDisposables[characteristic] {
            disposable.dispose()
            isNotifyDisposables[characteristic] = nil
        }
    }
    
    /// 它触发特征的通知启动. 并一次性的订阅
    private func observeValueUpdateAndSetNotification(for characteristic: Characteristic) {
        if let _ = notificationDisposables[characteristic] { return }
        let disposable = characteristic.observeValueUpdateAndSetNotification().subscribe(
            onNext: { [weak self] (characteristic) in
                guard let strong = self else { return }
                strong.updatedValueAndNotificationSubject.onNext(characteristic)
            },
            onError: { _ in }
        )
        notificationDisposables[characteristic] = disposable
    }
    
    private func disposeNotification(for characteristic: Characteristic) {
        if let disposable = notificationDisposables[characteristic] {
            disposable.dispose()
            notificationDisposables[characteristic] = nil
        }
    }
    
        
    /// 当您观察到从外围设备断开连接时，您希望确保对.next和.error事件都采取了操作。
    /// 例如，当你的设备BluetoothState.poweredOff，你将收到一个.error事件.
    private func observeDisconnect(for peripheral: Peripheral) {
        if let _ = disconnectinDisposables[peripheral] { return }
        let disposable = centralManager.observeDisconnect(for: peripheral).subscribe(onNext: { [unowned self] (peripheral, reason) in
            self.disconnectionSubject.onNext((peripheral, reason))
        })
        disconnectinDisposables[peripheral] = disposable;
    }
    
    private func disposeDisconnect(for peripheral: Peripheral) {
        if let disposable = disconnectinDisposables[peripheral] {
            disposable.dispose()
            disconnectinDisposables[peripheral] = nil
        }
    }
    
    /// 监听名称发生更改.
    private func observeUpdateName(for peripheral: Peripheral) {
        let disposable = peripheral.observeNameUpdate().subscribe(
            onNext: { [unowned self] (peripheral, name) in
                guard let hasBindPeripheral = self.bindPeripheral,
                      hasBindPeripheral.uuid == peripheral.identifier.uuidString,
                      let newName = peripheral.name else { return }
                hasBindPeripheral.updateName(newName)
                self.updatedNameSubject.onNext((peripheral, name))
            },
            onError: { _ in }
        )
        updateNameDisposables[peripheral] = disposable
    }
    
    
    private func disposeUpdateName(for peripheral: Peripheral) {
        if let disposable = updateNameDisposables[peripheral] {
            disposable.dispose()
            updateNameDisposables[peripheral] = nil
        }
    }
}


// MARK: - Error

public enum BandServiceError: Error, CustomStringConvertible {
    case redundantStateChange
    case establishConnectionTimeout
    case scanningStop
    case delete

    public var description: String {
        switch self {
        case .redundantStateChange: return "Redundant state change."
        case .establishConnectionTimeout: return "Establish connection timeout."
        case .scanningStop: return "The end of the scanning."
        case .delete:return "Remove the device."
        }
    }
}

extension BluetoothError {
    init?(state: BluetoothState) {
        switch state {
        case .unsupported: self = .bluetoothUnsupported
        case .unauthorized: self = .bluetoothUnauthorized
        case .poweredOff: self = .bluetoothPoweredOff
        case .unknown: self = .bluetoothInUnknownState
        case .resetting: self = .bluetoothResetting
        default: return nil
        }
    }
}
