//
//  SRPeripheral.swift
//  CTBandKit
//
//  Created by Mac on 2020/4/20.
//

import Foundation
import RxBluetoothKit

/// 表示被搜索到的外设实例
/// - 扫描到的外设——包含它的 advertismen t数据、rssi 和 periphera 本身
/// - 已连接到 CentralManager 的外设 —— 只有 periphera 本身
/// 为了执行进一步的操作，可以使用“peripheral”实例变量，保持本地连接
public class SRPeripheral {
    
    // MARK: - Public
    public lazy var peripheralName : String = { return advertisementData.localName ?? (peripheral.name ?? "Unknown Name") } ()
    
    public var rssiText : String { return "rssi: \(rssi)" }
    
    public var rssiValue : Int { return rssi.intValue }
    
    public var uuid: String { return peripheral.identifier.uuidString }
    
    /// 是否是 duf 模式设备
    public var isDuf: Bool {
        let str = peripheralName
        if str.hasPrefix("DFU_") { return true }
        return false
    }
    
    /// dfu 设备名称是否带有 Model 号
    /// 通过 含有 “_” 的数量判断，当 只有 1个 时 含有 Model 号
    /// 旧的固件 dfu 设备名称都含有 2 个 “_”
    public var dfuMode: String? {
        guard isDuf else {  return nil }
        let str = peripheralName
        var count = 0
        var idx = 0
        str.forEach { (e) in
            if count == 0 { idx += 1 }
            if e=="_" { count += 1 }
        }
        /// 下划线的数量
        if count == 1 {
            let modelStr:String = String(str.suffix(str.count - idx))
            return modelStr
        }
        return nil
    }


    // MARK: - Internal
    /// `Peripheral` 实例, 这允许执行进一步的蓝牙操作.
    let peripheral: Peripheral

    /// 扫描peripheral的广告数据
    let advertisementData: AdvertisementData

    /// RSSI value.
    let rssi: NSNumber
    
    
    init(peripheral: Peripheral, advertisementData: AdvertisementData, rssi: NSNumber) {
        self.peripheral = peripheral
        self.advertisementData = advertisementData
        self.rssi = rssi
    }
    
    convenience init(peripheral: ScannedPeripheral) {
        self.init(peripheral: peripheral.peripheral, advertisementData: peripheral.advertisementData, rssi: peripheral.rssi)
    }
    
    convenience init(peripheral: Peripheral) {
        self.init(peripheral: peripheral, advertisementData: AdvertisementData(advertisementData: [String : Any]()), rssi: NSNumber(value: 0))
    }
    
    
}

extension SRPeripheral: Comparable {

    public static func < (lhs: SRPeripheral, rhs: SRPeripheral) -> Bool {
        return (lhs.rssi.intValue < rhs.rssi.intValue)
    }
    
    public static func == (lhs: SRPeripheral, rhs: SRPeripheral) -> Bool {
        return lhs.peripheral == rhs.peripheral
    }
}

extension SRPeripheral: CustomStringConvertible {
    public var description: String {
        return "\(peripheralName) - \(uuid)"
    }
}
