
//
//  SRCalendarUtils.swift
//  CTFit
//
//  Created by Shirajo on 2018/8/7.
//  Copyright © 2018年 Shirajo. All rights reserved.
//

import Foundation

class SRCalendarUtils: NSObject  {
    
    // MARK: 常量
    static let kSecondsOneMinutes   = 60;
    static let kSecondsOneHour      = 60 * SRCalendarUtils.kSecondsOneMinutes;
    static let kSecondsOneDay       = 24 * SRCalendarUtils.kSecondsOneHour;
    
    static let kShortWeekSymbols: [String] = Calendar.autoupdatingCurrent.shortStandaloneWeekdaySymbols;
    
    static let kVeryShortWeekdaySymbols: [String] = Calendar.autoupdatingCurrent.veryShortWeekdaySymbols;

    
    static let kDateComponentAll: Set = [Calendar.Component.year, Calendar.Component.month, Calendar.Component.day,
                                          Calendar.Component.hour, Calendar.Component.minute, Calendar.Component.second,
                                          Calendar.Component.nanosecond, Calendar.Component.weekday];
    
    static let kDateComponentTime: Set = [Calendar.Component.hour, Calendar.Component.minute, Calendar.Component.second,Calendar.Component.nanosecond];
    
    static let calendar: Calendar = Calendar(identifier: .gregorian)

    
    // MARK: - 提供方便使用的方法
    
    static func date(withHour hourOfDay: Int, minute: Int = 0) -> Date {
        var dateComponents: DateComponents = DateComponents();
        dateComponents.hour = hourOfDay;
        dateComponents.minute = minute;
        return calendar.date(from: dateComponents)!;
    }
    
    static func date(withYear year: Int, month: Int, day: Int,
                     hour: Int = 0, minute: Int = 0, second: Int = 0, nanosecond: Int = 0) -> Date {
        var dateComponents: DateComponents = DateComponents();
        dateComponents.year = year;
        dateComponents.month = month;
        dateComponents.day = day;
        dateComponents.hour = hour;
        dateComponents.minute = minute;
        dateComponents.second = second;
        dateComponents.nanosecond = nanosecond;
        return calendar.date(from: dateComponents)!;
    }
    
    /// 字符串转 Date
    static func date(from dateStr: String, format: String) -> Date {
        let dateFormatter: DateFormatter = DateFormatter();
        dateFormatter.dateFormat = format;
        return dateFormatter.date(from: dateStr)!;
    }
    
    
    /// Component
    static func components(_ components: Set<Calendar.Component>, from date1: Date, to date2: Date) -> DateComponents {
        return calendar.dateComponents(components, from: date1, to: date2)
    }
    
    static func components(ofDate date: Date) -> DateComponents {
        let components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        return components;
    }
    
    static func year(ofDate date: Date) -> Int {
        let components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        return components.year!;
    }
    
    static func month(ofDate date: Date) -> Int {
        let components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        return components.month!;
    }
    
    static func day(ofDate date: Date) -> Int {
        let components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        return components.day!;
    }
    
    static func weekDay(ofDate date: Date) -> Int {
        let components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        return components.weekday!;
    }
    
    static func hour(ofDate date: Date) -> Int {
        let components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        return components.hour!;
    }
    
    static func minute(ofDate date: Date) -> Int {
        let components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        return components.minute!;
    }
    
    /// 清除 日时分秒
    static func monthDate(for date: Date) -> Date {
        var components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        components.day = 1;
        components.hour = 0;
        components.minute = 0;
        components.second = 0;
        components.nanosecond = 0;
        return calendar.date(from: components)!;
    }
    
    /// 清除 时分秒
    static func dayDate(for date: Date) -> Date {
        var components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        components.hour = 0;
        components.minute = 0;
        components.second = 0;
        components.nanosecond = 0;
        return calendar.date(from: components)!;
    }
    
    /// 清除 分秒
    static func hourDate(for date: Date) -> Date {
        var components: DateComponents = calendar.dateComponents(kDateComponentAll, from: date);
        components.minute = 0;
        components.second = 0;
        components.nanosecond = 0;
        return calendar.date(from: components)!;
    }
    
    /// 是否为当天
    static func isToday(_ date: Date) -> Bool {
        return calendar.isDateInToday(date);
    }
    
    static func stringFormatWithHour(_ hour: Int, _ minute: Int, _ hourly: SRHourlyType) -> String {
        if hourly == .h12 {
            return String(format: "%02ld:%02ld", CLong(hour),CLong(minute))
        } else {
            var symbol: String = calendar.amSymbol
            if hour >= 12 { symbol = calendar.pmSymbol }
            var newHour: Int = hour%12
            if newHour == 0 { newHour = 12 }
            return String(format: "%02ld:%02ld%@", CLong(newHour),CLong(minute), symbol)
        }
    }
    
    static func systemHourly() -> SRHourlyType {
        if let formatStringForHours: String = DateFormatter.dateFormat(fromTemplate: "j", options: 0, locale: Locale.current),
            let _: Range = formatStringForHours.range(of: "a") {
            return .h12
        }
        return .h24
    }

    static func systemHourlyC() -> SRHourlyType {
        let locale: Locale = Locale.autoupdatingCurrent
        var calendar: Calendar = Calendar.autoupdatingCurrent
        calendar.locale = locale
        let amSymbol: String = calendar.amSymbol
        let pmSymbol: String = calendar.pmSymbol
        let dateStr: String = Date().description(with: locale)
        for symbol in [amSymbol,pmSymbol] {
            if dateStr.range(of: symbol) != nil {
                return .h12
            }
        }
        return .h24
    }
    
}

extension SRCalendarUtils {
    
    static let dateFormatter :DateFormatter = {
        let fm = DateFormatter();
        fm.dateFormat = "yyyy-MM-dd";
        return fm;
    }()
    
    static func dateStr(timeInterval: TimeInterval) -> String {
        dateFormatter.string(from: Date(timeIntervalSince1970: timeInterval))
    }
    
    static func dateStr(date: Date) -> String {
        dateFormatter.string(from: date)
    }
    
    static let datetimeFormatter :DateFormatter = {
        let fm = DateFormatter();
        fm.dateFormat = "yyyy-MM-dd HH:mm:ss.SSS";
        return fm;
    }()
    
    static func datetimeStr(date: Date) -> String {
        datetimeFormatter.string(from: date)
    }
    
    static let timeFormatter :DateFormatter = {
        let fm = DateFormatter();
        fm.dateFormat = "HH:mm"
        return fm;
    }()
    
    static func timeStr(date: Date) -> String {
        timeFormatter.string(from: date)
    }
    
    static func timeStr(timeInterval: TimeInterval) -> String {
        timeFormatter.string(from: Date(timeIntervalSince1970: timeInterval))
    }
    
    static func minutesOffset(from timeStart: TimeInterval, to timeEnd: TimeInterval) -> Int {
        let offset = (timeEnd - timeStart)/60
        return max(0, NSNumber(value: offset).intValue)
    }
}

extension SRCalendarUtils {
    //MARK: - 手机时间戳 与 手环时间戳转换
    
    static let calendarGMT: Calendar = {
        var cal = Calendar(identifier: .gregorian)
        cal.locale = Locale.current
        cal.timeZone = TimeZone(secondsFromGMT: 0) ?? .current
        return cal
    } ()
    
    static let timeZone: TimeZone = {
        TimeZone.current
    } ()

    static let timeZoneOffset: TimeInterval = {
        TimeInterval(timeZone.secondsFromGMT(for: Date()))
    } ()
    
    static func bandTime(fromPhone time: TimeInterval) -> TimeInterval {
        return time + timeZoneOffset
    }
    
    static func phoneTime(fromBand time: TimeInterval) -> TimeInterval {
        return time - timeZoneOffset
    }
}
