//
//  FileListViewController.swift
//  FileBrowser
//
//  Created by Roy Marmelstein on 12/02/2016.
//  Copyright © 2016 Roy Marmelstein. All rights reserved.
//

import Foundation
import UIKit

class FileListViewController: UIViewController {

    let collation = UILocalizedIndexedCollation.current()
    
    /// Data
    var didSelectFile: ((FBFile) -> ())? 
    var files = [FBFile]()
    var initialPath: URL?
    let parser = FileParser.sharedInstance
    let previewManager = PreviewManager()
    var sections: [[FBFile]] = []
    var allowEditing: Bool = false
    
    // Search controller
    var filteredFiles = [FBFile]()


    lazy var tableView: UITableView = {
        let view = UITableView(frame: UIScreen.main.bounds, style: .plain)
        view.dataSource = self
        view.delegate = self
        return view
    }()
    
    lazy var searchController: UISearchController = {
        let searchController = UISearchController(searchResultsController: nil)
        searchController.dimsBackgroundDuringPresentation = false
        searchController.searchBar.searchBarStyle = .minimal
        searchController.searchBar.isTranslucent = false
        return searchController
    }()
    
    lazy var searchBar: UISearchBar  = {
        let bar = UISearchBar()
        bar.bounds = CGRect(x: 0, y: 0, width: UIScreen.main.bounds.self.width, height: 57)
        bar.searchBarStyle = .default
        //bar.translatesAutoresizingMaskIntoConstraints = false
        return bar
    }()
    
    lazy var headerView: UITableViewHeaderFooterView = {
        let view = UITableViewHeaderFooterView(frame: CGRect(x: 0, y: 0, width: UIScreen.main.bounds.size.width, height: 88))
        view.contentView.backgroundColor = .lightGray
        return view
    }()
    
        
    //MARK: Lifecycle
    
    convenience init (initialPath: URL) {
        self.init(initialPath: initialPath, showCancelButton: true)
    }
    
    convenience init (initialPath: URL, showCancelButton: Bool) {
        self.init()
        self.title = initialPath.lastPathComponent

        // Set initial path
        self.initialPath = initialPath
        // dismiss barItem
        if showCancelButton { showDismissItem() }

        // Set search controller delegates
        searchController.searchBar.delegate = self
        searchController.searchResultsUpdater = self
        searchController.delegate = self
    }
    
    
    //MARK: UIViewController
    
    override func viewWillLayoutSubviews() {
        super.viewWillLayoutSubviews()
        tableView.frame = self.view.bounds
    }
    
    override func viewDidLoad() {
        navigationController?.view.backgroundColor = UIColor.dark(.black, .white)
        if #available(iOS 11.0, *) {
            navigationController?.navigationItem.hidesSearchBarWhenScrolling = true
        } else {
            searchController.hidesNavigationBarDuringPresentation = false
            // Fallback on earlier versions
        }
        
        prepareData()
        self.view.addSubview(tableView)
        // Register for 3D touch
        self.registerFor3DTouch()


        // Set search bar
        if #available(iOS 11.0, *) {
            navigationItem.searchController = searchController
        } else {

        }
        
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
    }
    
    func showDismissItem() {
        let item = UIBarButtonItem(barButtonSystemItem: .cancel, target: self, action: #selector(onDismiss))
        self.navigationItem.rightBarButtonItem = item
    }
    
    @objc func onDismiss() {
        self.dismiss(animated: true, completion: nil)
    }
    
    func prepareData() {
        if let initialPath = initialPath {
            files = parser.filesForDirectory(initialPath)
            indexFiles()
        }
    }
    
    //MARK: Data
    
    func indexFiles() {
        let selector: Selector = #selector(getter: FBFile.displayName)
        sections = Array(repeating: [], count: collation.sectionTitles.count)
        if let sortedObjects = collation.sortedArray(from: files, collationStringSelector: selector) as? [FBFile]{
            for object in sortedObjects {
                let sectionNumber = collation.section(for: object, collationStringSelector: selector)
                sections[sectionNumber].append(object)
            }
        }
    }
    
    func fileForIndexPath(_ indexPath: IndexPath) -> FBFile {
        var file: FBFile
        if searchController.isActive {
            file = filteredFiles[(indexPath as NSIndexPath).row]
        }
        else {
            file = sections[(indexPath as NSIndexPath).section][(indexPath as NSIndexPath).row]
        }
        return file
    }
    
    func filterContentForSearchText(_ searchText: String) {
        filteredFiles = files.filter({ (file: FBFile) -> Bool in
            return file.displayName.lowercased().contains(searchText.lowercased())
        })
        tableView.reloadData()
    }

}

extension UIColor {
    
    /// rgbValue -> UIColor
    convenience init(rgb value: UInt32, alpha: CGFloat = 1) {
        let red     = CGFloat((value & 0xFF0000) >> 16)/255.0
        let green   = CGFloat((value & 0xFF00) >> 8)/255.0
        let blue    = CGFloat((value & 0xFF) >> 0)/255.0
        self.init(red: red, green: green, blue: blue, alpha: alpha)
    }
    
    /// Hex String -> UIColor
    convenience init(hexString: String, alpha: CGFloat = 1) {
        let hexString = hexString.trimmingCharacters(in: .whitespacesAndNewlines)
        let scanner = Scanner(string: hexString)
        if hexString.hasPrefix("#") {
            scanner.scanLocation = 1
        }
        var color: UInt32 = 0
        scanner.scanHexInt32(&color)
        self.init(rgb: color, alpha: alpha)
    }
     
    /// UIColor -> Hex String
    var hexString: String? {
        var red: CGFloat = 0
        var green: CGFloat = 0
        var blue: CGFloat = 0
        var alpha: CGFloat = 0
        let multiplier = CGFloat(255.999999)
        guard self.getRed(&red, green: &green, blue: &blue, alpha: &alpha) else { return nil }
        if alpha == 1.0 {
            return String( format: "#%02lX%02lX%02lX", Int(red * multiplier), Int(green * multiplier), Int(blue * multiplier) )
        } else {
            return String(format: "#%02lX%02lX%02lX%02lX", Int(red * multiplier), Int(green * multiplier), Int(blue * multiplier), Int(alpha * multiplier) )
        }
    }
    
    /// iOS 13 暗黑模式颜色适配
    static func dark(_ darkColor: UIColor, _ defaultColor:UIColor) -> UIColor {
        if #available(iOS 13.0, *) {
            let colorDynamic = UIColor.init { (traitCollection) -> UIColor in
                if (traitCollection.userInterfaceStyle == .dark) {
                    return darkColor
                } else {
                    return defaultColor
                }
            }
            return colorDynamic
        }
        return defaultColor
    }
}
