//
//  BaseNavigation.swift
//  BandKit
//
//  Created by Mac on 2020/5/26.
//

import Foundation
import NavigationExtension

open class BaseNavigation: UINavigationController {
    
    override open func viewDidLoad() {
        super.viewDidLoad()
        self.jz_navigationBarTransitionStyle = JZNavigationBarTransitionStyleSystem;
        self.jz_fullScreenInteractivePopGestureEnabled = true;
               
        navigationBar.isTranslucent = false;
        navigationBar.barStyle = .black;
        navigationBar.barTintColor = Colors.NavigationBar.barTint;
        navigationBar.tintColor = Colors.NavigationBar.tint;
        navigationBar.titleTextAttributes = [NSAttributedString.Key.foregroundColor: Colors.NavigationBar.foreground]
        navigationBar.setBackgroundImage(UIImage(), for: .default)
        navigationBar.shadowImage = navBarShadowImage()
    }
}

extension BaseNavigation {
    func navBarShadowImage() -> UIImage? {
        let scrWidth = UIScreen.main.bounds.size.width
        UIGraphicsBeginImageContextWithOptions(CGSize(width: scrWidth, height: 0.5), false, 0)
        let path = UIBezierPath.init(rect: CGRect.init(x: 0, y: 0, width: scrWidth, height: 0.5))
        /// 自定义NavigationBar分割线颜色
        Colors.NavigationBar.separator.setFill()
        path.fill()
        let image = UIGraphicsGetImageFromCurrentImageContext()
        UIGraphicsEndImageContext()
        return image
    }
}

extension UIColor {
    
    /// rgbValue -> UIColor
    convenience init(rgb value: UInt32, alpha: CGFloat = 1) {
        let red     = CGFloat((value & 0xFF0000) >> 16)/255.0
        let green   = CGFloat((value & 0xFF00) >> 8)/255.0
        let blue    = CGFloat((value & 0xFF) >> 0)/255.0
        self.init(red: red, green: green, blue: blue, alpha: alpha)
    }
    
    /// Hex String -> UIColor
    convenience init(hexString: String, alpha: CGFloat = 1) {
        let hexString = hexString.trimmingCharacters(in: .whitespacesAndNewlines)
        let scanner = Scanner(string: hexString)
        if hexString.hasPrefix("#") {
            scanner.scanLocation = 1
        }
        var color: UInt32 = 0
        scanner.scanHexInt32(&color)
        self.init(rgb: color, alpha: alpha)
    }
     
    /// UIColor -> Hex String
    var hexString: String? {
        var red: CGFloat = 0
        var green: CGFloat = 0
        var blue: CGFloat = 0
        var alpha: CGFloat = 0
        let multiplier = CGFloat(255.999999)
        guard self.getRed(&red, green: &green, blue: &blue, alpha: &alpha) else { return nil }
        if alpha == 1.0 {
            return String( format: "#%02lX%02lX%02lX", Int(red * multiplier), Int(green * multiplier), Int(blue * multiplier) )
        } else {
            return String(format: "#%02lX%02lX%02lX%02lX", Int(red * multiplier), Int(green * multiplier), Int(blue * multiplier), Int(alpha * multiplier) )
        }
    }
    
    /// iOS 13 暗黑模式颜色适配
    static func darkMode(_ darkColor: UIColor, _ defaultColor:UIColor) -> UIColor {
        if #available(iOS 13.0, *) {
            let colorDynamic = UIColor.init { (traitCollection) -> UIColor in
                if (traitCollection.userInterfaceStyle == .dark) {
                    return darkColor
                } else {
                    return defaultColor
                }
            }
            return colorDynamic
        }
        return defaultColor
    }
}

struct Colors {
    
    private static func rgb(_ value: UInt32, _ alpha: CGFloat = 1.0)-> UIColor {
        UIColor(rgb: value, alpha: alpha)
    }
    private static func rgb(hexString: String, alpha: CGFloat = 1.0)-> UIColor {
        UIColor(hexString: hexString, alpha: alpha)
    }
    private static func rgb(value: UInt32, alpha: CGFloat = 1.0)-> UIColor {
        UIColor(rgb: value, alpha: alpha)
    }
    private static func darkMode(_ darkColor: UIColor, _ defaultColor:UIColor) -> UIColor {
        UIColor.darkMode(darkColor, defaultColor)
    }
        
    static let black: UIColor = rgb(0x000000)
    static let white: UIColor = rgb(0xFFFFFF)
    static let dark: UIColor = rgb(0x2F2F2F)
    static let gray: UIColor = rgb(0xAAAAAA)
    static let blue: UIColor = rgb(0x33FFFF)
    static let red: UIColor = rgb(0xFF0000)
    
    struct NavigationBar {
        static let barTint: UIColor = darkMode(black, white)
        static let tint: UIColor = darkMode(white, black)
        static let foreground: UIColor = darkMode(white, black)
        static let separator: UIColor = darkMode(dark, gray)
    }
    
    struct Base {
        static let background: UIColor = darkMode(black, white)
        static let title: UIColor = darkMode(white, black)
        static let description: UIColor = darkMode(gray, dark)
    }
}
