//
//  NELocationManager.swift
//  CTFit
//
//  Created by Mac on 2020/6/12.
//  Copyright © 2020 shirajo. All rights reserved.
//


import UIKit
import CoreLocation

public class NELocationManager: NSObject {
    
    typealias AuthorizationClosure = (CLAuthorizationStatus) -> Void
    
    typealias UpdateHeadingClosure = (CLHeading) -> Void
    
    typealias UpdateLocationClosure = (CLLocation) -> Void
    
    var didChangeAuthorizatio: AuthorizationClosure? = nil

    var didUpdateHeading: UpdateHeadingClosure? = nil
       
    var didUpdateLocation: UpdateLocationClosure? = nil

    static let shared = NELocationManager()
    
    override init() {
        super.init()
        
        self.locationManager.delegate = self
    }
    
    convenience init(didUpdateHeading: @escaping UpdateHeadingClosure, didUpdateLocation: @escaping UpdateLocationClosure) {
        self.init()
        self.didUpdateHeading = didUpdateHeading
        self.didUpdateLocation = didUpdateLocation
    }
    
    
    // MARK: private filed
    
    private var locationManager: CLLocationManager = {
        let manager = CLLocationManager();
        manager.requestAlwaysAuthorization(); // 请求权限验证
        manager.desiredAccuracy = kCLLocationAccuracyBest; // 尽可能达到最佳的准确性
        manager.distanceFilter = 2; // 指定最小更新距离(单位为米)。
        manager.headingFilter = 1;  // 更新方向角度的最小偏差(箭头表示用户当前pos)
        manager.activityType = .fitness
        manager.pausesLocationUpdatesAutomatically = false // 指定可能时可自动暂停位置更新
        if #available(iOS 9.0, *) { // 开启后台定位
            manager.allowsBackgroundLocationUpdates = true;
        }
        return manager;
    }()
    
    private var hasPermissionForLocation: Bool {
        if NELocationManager.hasPermissionForLocation(),
            !(NELocationManager.runningInBackground() && NELocationManager.hasPermissionWhenInUseForLocation()) {
            return true
        }
        return false
    }
    
    // MARK: Public method
    func requestLocation() {
        if #available(iOS 9.0, *) {
            locationManager.requestLocation()
        }
    }

    func startUpdatingLocation() {
        locationManager.startUpdatingLocation()
    }
    
    func stopUpdatingLocation() {
        locationManager.stopUpdatingLocation()
    }
    
    func startUpdatingHeading() {
        locationManager.startUpdatingHeading()
    }
    
    func stopUpdatingHeading() {
        locationManager.stopUpdatingHeading()
    }

}

// MARK: -  LatestLocation

extension NELocationManager {
    
    static let  kLatestLocationKey: String = "kBandLatestLocationKey"

    static var latestLocation: CLLocationCoordinate2D? {
        get {
            if _latestLocation != nil { return _latestLocation! }
            if let cache = UserDefaults.standard.dictionary(forKey: kLatestLocationKey),
                let latitude: Double = cache["latitude"] as? Double,
                let longitude: Double = cache["longitude"] as? Double {
                _latestLocation = CLLocationCoordinate2D(latitude: latitude, longitude: longitude)
            }
            return _latestLocation
        } set {
            _latestLocation = newValue
        }
    }
    
    private static var _latestLocation: CLLocationCoordinate2D? {
        didSet {
            guard _latestLocation != oldValue else { return }
            let dictionary = ["latitude": _latestLocation?.latitude,"longitude": _latestLocation?.longitude]
            UserDefaults.standard.set(dictionary, forKey: kLatestLocationKey)
            UserDefaults.standard.synchronize()
        }
    }
}

extension CLLocationCoordinate2D: Equatable {
    public static func == (lhs: CLLocationCoordinate2D, rhs: CLLocationCoordinate2D) -> Bool {
        return (lhs.latitude == rhs.latitude && lhs.longitude == rhs.longitude)
    }
}



// MARK: - CLLocationManagerDelegate

extension NELocationManager: CLLocationManagerDelegate {
    /// 改变授权状态
    public func locationManager(_ manager: CLLocationManager, didChangeAuthorization status: CLAuthorizationStatus) {
        // NELog.v("status: \(status.rawValue)")
        guard let block = didChangeAuthorizatio else { return }
        block(status)
    }
    
    /// 定位失败
    public func locationManager(_ manager: CLLocationManager, didFailWithError error: Error) {
        // NELog.v("error: \(error)")
    }
    
    /// 方向更新
    public func locationManager(_ manager: CLLocationManager, didUpdateHeading newHeading: CLHeading) {
        // NELog.v("newHeading: \(newHeading)")
        guard let block = didUpdateHeading else { return }
        block(newHeading)
    }
    
    /// 用户位置更新
    public func locationManager(_ manager: CLLocationManager, didUpdateLocations locations: [CLLocation]) {
        // NELog.v("didUpdateLocations: \(locations)")
        guard let newLocation = locations.first else { return }
        NELocationManager.latestLocation = newLocation.coordinate
        guard let block = didUpdateLocation  else { return }
        block(newLocation)
    }
}


// MARK: - 定位权限

extension NELocationManager {
    ///: - 可以请求到位置判断
    public static func canRequestLocation() -> Bool{
        if hasPermissionAlwaysForLocation() { return true }
        if hasPermissionWhenInUseForLocation(), !runningInBackground() { return true }
        return false
    }
    
    ///: - 后台运行判断
    public static func runningInBackground() -> Bool {
        return (UIApplication.shared.applicationState == .background);
    }
    
    ///: - 判断当前定位是否可用，不可用弹窗提示
    public static func locationServicesEnabled () -> Bool {
        if !CLLocationManager.locationServicesEnabled() {
            print("设置打开定位授权")
            return false
        }
        return true
    }
    
    ///: - 同意了定位权限
    public static func hasPermissionForLocation() -> Bool {
        let enable = CLLocationManager.locationServicesEnabled();
        if !enable {
            return false
        }
        let status = CLLocationManager.authorizationStatus()
        if status == .authorizedAlways || status == .authorizedWhenInUse {
            return true;
        }
        return false
    }
    
    ///: - Always 权限
    public static func hasPermissionAlwaysForLocation() -> Bool {
        let enable = CLLocationManager.locationServicesEnabled();
        if !enable {
            return false
        }
        let status = CLLocationManager.authorizationStatus()
        if status == .authorizedAlways { return true; }
        return false
    }
    
    ///: - WhenInUse 权限
    public static func hasPermissionWhenInUseForLocation() -> Bool {
        let enable = CLLocationManager.locationServicesEnabled();
        if !enable {
            return false
        }
        let status = CLLocationManager.authorizationStatus()
        if status == .authorizedWhenInUse { return true; }
        return false
    }
    
    ///: - 拒绝定位权限
    public static func hasDeniedUseForLocation() -> Bool {
        let enable = CLLocationManager.locationServicesEnabled();
        if !enable {
            return false
        }
        let status = CLLocationManager.authorizationStatus()
        if status == .denied { return true; }
        return false
    }
    ///: - 用户还没有对这个应用程序作出选择
    public static func hasNoDeniedUseForLocation() -> Bool {
        let enable = CLLocationManager.locationServicesEnabled();
        if !enable {
            return false
        }
        let status = CLLocationManager.authorizationStatus()
        if status == .notDetermined { return true; }
        return false
    }
}




