//
//  NENetworkingManger.swift
//  CTFit
//
//  Created by Mac on 2020/7/1.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation
import Alamofire
import CoreLocation

public class NENetworkingManger {
    
    // MARK: - Private Method
    private static func jsonString(from dictionary: [String: Any]?, prettyPrint: Bool = false) -> String? {
        if let anyObject = dictionary {
            if JSONSerialization.isValidJSONObject(anyObject) {
                do {
                    let jsonData: Data
                    var options: JSONSerialization.WritingOptions = []
                    if #available(iOS 11.0, *) { options.insert(.sortedKeys) }
                    if prettyPrint { options.insert(.prettyPrinted) }
                    jsonData = try JSONSerialization.data(withJSONObject: anyObject, options: options)
                    return String(data: jsonData, encoding: .utf8)
                } catch {
                    NELog.w("\(error)")
                }
            }
        }
        return nil
    }


    private static func dictionary(from text: String) -> [String: Any]? {
        if let data = text.data(using: String.Encoding.utf8) {
            do {
                let anyObject = try JSONSerialization.jsonObject(with: data, options: [JSONSerialization.ReadingOptions(rawValue: 0)])
                return  anyObject as? [String: Any]
            } catch {
                NELog.w("\(error)")
            }
        }
        return nil
    }
    
    private static let sessionManager: SessionManager = {
        /// 支持后台请求
        let info = Bundle.main.infoDictionary
        let bundleId = info?["CFBundleIdentifier"] as? String
        let lab = "\(bundleId ?? "com.bandkirpro").sessionmanager";
        let configuration = URLSessionConfiguration.background(withIdentifier: lab)
        configuration.httpAdditionalHeaders = SessionManager.defaultHTTPHeaders
        configuration.timeoutIntervalForRequest = 60
        return SessionManager(configuration: configuration)
    }()
    
    // MARK: - HTTP Get请求
    /// HTTP 请求
    /// - Parameters:
    ///   - url: 请求路劲
    ///   - parameters: 参数
    ///   - completionHandler: 结果回调
    private static func request(_ url: String, _ parameters: Parameters?, _ completionHandler: @escaping (DataResponse<String>) -> Void) -> DataRequest {
        let parametersString = jsonString(from: parameters) ?? "{}"
        return sessionManager.request(url, method: .get, parameters: parameters).responseString(completionHandler: { (response) in
            switch response.result {
            case .success(let text):
                NELog.i("""
                url: \(url)
                param: \(parametersString)
                reponse: \(text)
                """)
            case .failure(let error):
                NELog.w("""
                url: \(url)
                param: \(parametersString)
                error: \(error)
                """)
            }
            completionHandler(response)
            }
        )
    }
    
    // MARK: - 文件下载
    /// 文件下载
    /// - Parameters:
    ///   - url: 下载路劲
    ///   - filePathURL: 文件存储路劲
    ///   - parameters: 请求参数
    ///   - completionHandler: 结果回调
    private static func download(_ url: String, _ filePathURL: URL, _ parameters: Parameters?, _ completionHandler: @escaping (DownloadResponse<String>) -> Void) -> DownloadRequest {
        /// 请求参数
        let parametersString = jsonString(from: parameters) ?? "{}"
        /// 文件存储路劲
        let destination: DownloadRequest.DownloadFileDestination = { temporaryURL,_ in
            let fileURL = filePathURL.appendingPathComponent(URL(fileURLWithPath: url).lastPathComponent)
            return (fileURL, [.createIntermediateDirectories, .removePreviousFile])
        }
        /// 请求
        let downloadRequest = sessionManager.download(url, method: .get, parameters: parameters, to: destination)
        /// String响应
        downloadRequest.responseString { (response) in
            switch response.result {
            case .success(_):
                NELog.i("""
                url: \(url)
                param: \(parametersString)
                """)
            case .failure(let error):
                NELog.w("""
                url: \(url)
                param: \(parametersString)
                error: \(error)
                """)
            }
            completionHandler(response)
        }
        
        return downloadRequest
    }
    
}

// MARK: - Class method
 
// MARK: 获取天气信息
extension NENetworkingManger {
    
    /// 天气请求
    private static var weaterRequest: DataRequest? = nil
    /// 获取天气形象
    /// - Parameters:
    ///   - loction: 定位位置
    ///   - failedCount: 连续错误次数
    ///   - completionHandler: 结果回调
    public static func weater(_ loction: CLLocationCoordinate2D, _ failedCount: Int,  _ completionHandler: @escaping (DataResponse<String>) -> Void) {
        if let hasRequest = weaterRequest { hasRequest.cancel() }
        let url = "http://api.shirajo.com/weather3.php"
        // let url = "http://test.shirajo.com/weather3.php"
        let location = loction
        let param: Parameters = ["lat": location.latitude, "lon": location.longitude, "t": failedCount]
        weaterRequest = request(url, param) { (response) in
            completionHandler(response)
        }
    }
}

// MARK: 固件更新
extension NENetworkingManger {

    /// : - 手表固件: 更新检测
    
    /// 手表固件更新检测请求
    private static var firmwareRequest: DataRequest? = nil
    
    /// 手表固件更新检测： 默认URL
    public static let firmwareDefaultUrl: String = "http://shirajo.com/ota2/ctupgrade/check.php"
    
    /// 手表固件更新检测
    /// - Parameters:
    ///   - url: 请求路劲
    ///   - model: 手表固件： Model 号
    ///   - ver: 手表固件：软件版本
    ///   - completionHandler: 结果回调
    public static func checkFirmwareUpdate(url: String = firmwareDefaultUrl, model: String, ver: Int,  completionHandler: @escaping (DataResponse<String>) -> Void) {
        let uuid = UIDevice.current.identifierForVendor?.uuidString ?? ""
        var version = ver
        #if DEBUG
        version = 0;
        #endif
        if let hasRequest = firmwareRequest { hasRequest.cancel() }
        let param: Parameters = ["devicemodel": model, "ver": version, "did": uuid]
        firmwareRequest = request(url, param) { (response) in
            completionHandler(response)
        }
    }
    
    /// : -  手表固件：软件包下载
    /// 存储下载文件路劲
    public static fileprivate(set) var downloadPathURL: URL = {
        return FileManager.default.urls(for: .documentDirectory, in: .userDomainMask).first!.appendingPathComponent("Download"/*, isDirectory: true*/)
    } ()
    
    /// 下载手表固件升级包
    /// - Parameters:
    ///   - url: 下载路劲
    ///   - completionHandler: 结果回调
    ///   - progressHandler: 下载进度回调
    public static func downloadFirmware(_ url: String, _ completionHandler: @escaping (DownloadResponse<String>) -> Void,
                                        _ progressHandler: @escaping Request.ProgressHandler) {
        let fileURL: URL = downloadPathURL
        let downloadRequest = self.download(url, fileURL, nil, completionHandler)
        /// 下载进度
        downloadRequest.downloadProgress { (progress) in
            NELog.i("Progress: \(progress.fractionCompleted)")
            progressHandler(progress)
        }
    }
}


// MARK: APNs Push
extension NENetworkingManger {
    /// apns 请求
    private static var apnsRequest: DataRequest? = nil
    /// 提交设备apns push 信息
    /// - Parameters:
    ///   - bundle: app唯一标识 bundle_id
    ///   - uuid: 手机唯一标识 uuid
    ///   - token: 远程通知deviceToken
    public static func apnsSubmit(_ bundle: String,  _ uuid: String,  _ token: String, _ completionHandler: @escaping (DataResponse<String>) -> Void) {
        if let hasRequest = apnsRequest { hasRequest.cancel() }
        let url = "http://apns.shirajo.com/Submit.php"
        var product = true
        #if DEBUG
        product = false
        #endif
        let param: Parameters = ["bundle": bundle, "uuid": uuid, "token": token, "product": product];
        apnsRequest = request(url, param) { (response) in
            completionHandler(response)
        }
    }
}
