//
//  NEWeatherInfo.swift
//  BandKit
//
//  Created by Mac on 2020/7/6.
//

import Foundation
import HandyJSON

private let userTestWeatherData: Bool = false

private let testWeatherInfo: String = {
    let currentDayDate =  NECalendarUtils.dayDate(for: Date())
    let currenthourtDate =  NECalendarUtils.hourDate(for: Date()) + 3600
    let daily_ts = UInt64(truncating: NSNumber(value: currentDayDate.timeIntervalSince1970))
    let hourly_ts = UInt64(truncating: NSNumber(value: currenthourtDate.timeIntervalSince1970))
    return """
    {
    "daily_ts":\(daily_ts),
    "hourly_ts":\(hourly_ts),
    "next":19,
    "hourly":[
    [801,21,1016,24,0,3,16],
    [801,21,1017,28,0,3,12],
    [801,20,1018,30,0,3,22],
    [803,19,1018,30,0,3,24],
    [804,18,1019,32,0,4,22],
    [804,18,1019,31,0,4,21],
    [804,18,1019,30,0,4,17],
    [804,17,1018,30,0,4,18],
    [804,17,1018,30,0,3,20],
    [804,17,1018,31,0,3,7],
    [804,17,1017,31,0,3,15],
    [804,16,1017,31,0,3,25],
    [804,16,1018,30,0,3,34],
    [804,16,1019,31,1,3,13],
    [804,16,1019,30,1,2,26],
    [803,17,1019,27,2,2,14],
    [804,19,1019,23,1,3,16],
    [804,19,1019,23,2,3,7],
    [801,20,1018,22,6,4,8],
    [801,22,1016,20,6,4,7],
    [801,23,1015,20,5,4,13],
    [801,23,1014,21,3,4,13],
    [801,23,1014,22,2,4,18],
    [801,23,1015,22,2,4,30]],
    "daily":[
    [23,15,804,29,399,1069],
    [23,15,803,30,399,1068],
    [25,15,802,29,400,1068],
    [24,18,803,36,400,1068],
    [22,18,804,40,401,1067],
    [25,16,801,34,402,1067],
    [22,18,804,31,402,1067],
    [23,18,804,37,403,1066],
    [25,18,803,43,404,1066],
    [27,20,803,41,404,1066],
    [27,21,804,42,405,1066],
    [27,21,804,51,405,1066],
    [29,22,803,48,406,1065],
    [30,23,801,54,407,1065],
    [31,24,801,52,407,1065],
    [31,24,801,53,407,1065],
    [29,23,801,51,407,1065],
    [29,23,801,51,407,1065]]}
    """
} ()

/// [0.0 ~ 1.0]
fileprivate func random() -> Float {
    return Float(drand48());
}

/// [min ~ max]
fileprivate func random(min: Int, max: Int) -> Int {
    return Int(truncatingIfNeeded: arc4random_uniform(UInt32(max - min + 1)) + UInt32(min))
}

public class NEWeatherInfo: NEModelProtocol {
    
    /// 洗一次查询天气的 小时间隔
    var next: Double = 24
    
    /// 未来天气预报 (小时) — 24 组
    /// 起始时间戳
    var hourly_ts: TimeInterval = 0
    /// (小时)
    /// 0:  天气ID
    /// 1:  当前温度
    /// 2:  大气压力
    /// 3:  相对湿度
    /// 4:  紫外线
    /// 5:  风速
    /// 6:  风向
    var hourly: [[Int]] = []
    
    /// 未来天气预报 (每天)  — 17 组
    /// 起始时间戳
    var daily_ts: TimeInterval = 0
    /// (每天)
    /// 0:  最高温
    /// 1:  最低温
    /// 2:  天气ID
    /// 3:  相对湿度
    /// 4:  日出(分钟数) , 如:  375=06:15
    /// 5:  日落(分钟数) , 如: 1117=18:37
    var daily: [[Int]] = []

    // MARK: - Initialize
    required public init() { }
}


// MARK: - Class to protocol method .set

extension NEWeatherInfo {
    
    // MARK: - Private
    
    static private(set) var shared: NEWeatherInfo = initeObj

    private static let WeatherInfoCacheKey = "WeatherInfoCacheKey"
    
    private static let initeObj: NEWeatherInfo = {
        if userTestWeatherData {
            let jsonString = testWeatherInfo
            guard let obj = deserialize(from: jsonString) else { return NEWeatherInfo() }
            observeAppNotify()
            return obj
        }
        guard let jsonString = UserDefaults.standard.string(forKey: WeatherInfoCacheKey),
            let obj = deserialize(from: jsonString) else { return NEWeatherInfo() }
        observeAppNotify()
        return obj
    }()
    
    /// 解析并缓存 服务器返回数据
    static func parseAndCache(_ serverJsonString: String) {
        guard let obj = deserialize(from: serverJsonString) else { return }
        UserDefaults.standard.set(serverJsonString, forKey: WeatherInfoCacheKey)
        UserDefaults.standard.synchronize()
        shared = obj
        NEWeatherInfo.updateCurrentSubject.onNext(currentHourItem)
    }
    
    static func parseAndCacheDebug() {
        parseAndCache(testWeatherInfo)
    }
    
    static var needUpdateQuery: Bool {
        let hourly_ts_random = TimeInterval(random(min: 0, max: 3599));
        let hourInterval = (Date().timeIntervalSince1970 - shared.hourly_ts - hourly_ts_random)/3600.0;
        if hourInterval > -8, hourInterval < shared.next { return false }
        return true
    }
}


extension NEWeatherInfo {
    
    public static var currentHourOffset: Int {
        let hourStartDate = Date(timeIntervalSince1970: shared.hourly_ts)
        let nowDate = Date()
        let components = NECalendarUtils.components([.hour], from: NECalendarUtils.hourDate(for: hourStartDate), to: NECalendarUtils.hourDate(for: nowDate))
        return components.hour!
    }

    public static var currentDayOffset: Int {
        let hourStartDate = Date(timeIntervalSince1970: shared.daily_ts)
        let nowDate = Date()
        let components = NECalendarUtils.components([.day], from: NECalendarUtils.dayDate(for: hourStartDate), to: NECalendarUtils.dayDate(for: nowDate))
        return components.day!
    }
}


// MARK: - Public

extension NEWeatherInfo {
    
    /// 未来天气预报 (小时) — 24 组
    /// 起始时间戳
    public static var hourly_ts: TimeInterval { shared.hourly_ts }

    /// 未来天气预报 (每天)  — 17 组
    /// 起始时间戳
    public static var daily_ts: TimeInterval { shared.daily_ts }
    
    /// 当前小时项
    public static var currentHourItem: NEWeatherHourItem? {
        var offsetHour = currentHourOffset

        /// 服务器新获取的数据，hourly_ts 可能大于当前时间一点点，即 offsetHour = -1
        /// 为了避免这种情况天气数据不能实时显示：
        /// hourly_ts 需减去一个小时，hourly数组左复制一组
        if offsetHour == -1 { offsetHour = 0 }
        
        guard offsetHour>=0, offsetHour < shared.hourly.count else { return nil }
        return NEWeatherHourItem.parse(shared.hourly[offsetHour])
    }
    
    /// 当天项
    public static var currentDayItem: NEWeatherDayItem? {
        let offsetDay = currentDayOffset
        guard offsetDay>=0, offsetDay < shared.daily.count else { return nil }
        return NEWeatherDayItem.parse(shared.daily[offsetDay])
    }
    
    /// 小时项 （>= 24 项）
    public static var hourItems: [NEWeatherHourItem]? {
        let items = shared.hourly.map { (array) -> NEWeatherHourItem? in NEWeatherHourItem.parse(array) }
        guard var validItems: [NEWeatherHourItem] = items as? [NEWeatherHourItem],
            validItems.count >= 24 else { return nil}
        
        /// 服务器新获取的数据，hourly_ts 可能大于当前时间一点点，即 offsetHour = -1
        /// 为了避免这种情况天气数据不能实时显示：
        /// hourly_ts 需减去一个小时，hourly数组左复制一组
        let offsetHour = currentHourOffset
        if offsetHour == -1 { validItems.insert(validItems[0].copy(), at: 0) }
        
        return validItems
    }
    
    /// 日项 （>= 15 项）
    public static var dayItems: [NEWeatherDayItem]? {
        let items = shared.daily.map { (array) -> NEWeatherDayItem? in NEWeatherDayItem.parse(array) }
        guard let validItems: [NEWeatherDayItem] = items as? [NEWeatherDayItem] ,
              validItems.count >= 15 else { return nil }
        return validItems
    }
}


/// 更新监听
import RxSwift
extension NEWeatherInfo {
    
    /// 更新数据
    public static var updatedCurrentOutput: Observable<NEWeatherHourItem?> {
        return updateCurrentSubject.asObservable()
    }
    
    private static let disposeBag = DisposeBag()
    private static let updateCurrentSubject = PublishSubject<NEWeatherHourItem?>()
    private static func observeAppNotify() {
        NEAppNotifyUtils.observable.subscribe(onNext: { type in
            switch type {
            case .willEnterForeground, .significantTimeChange:
                updateCurrentSubject.onNext(NEWeatherInfo.currentHourItem)
            default: break
            }
        }).disposed(by: disposeBag)
    }
}
