//
//  NEWeatherTask.swift
//  BandKit
//
//  Created by Mac on 2020/7/8.
//

import Foundation
import Alamofire
import CoreLocation
import Repeater

public class NEWeatherTask {
        
    init() {
        _ = locationManager
        
        locationManager.didChangeAuthorizatio = { [unowned self] (staus) in
            guard staus == .authorizedAlways || staus == .authorizedWhenInUse else { return }
            guard self.timer.state != .paused else { return }
            locationManager.startUpdatingLocation()
        }
        
        locationManager.didUpdateLocation = { [unowned self] (location) in
            DispatchQueue.main.async {
                locationManager.stopUpdatingLocation()
                NELog.i("didUpdateLocation: \(location)")
                NEWeatherTask.latestLocation = location.coordinate
                self.queryWeatherInfo()
            }
        }
    }
    
    private func start() {
        timer.pause()
        performQueryWeather()
        timer.start()
    }
    private func pause() {
        timer.pause()
    }
    
    // MARK: - Private
    
    private let interval: TimeInterval = 60*60
    
    private lazy var locationManager: NELocationManager = NELocationManager.shared

    private var isActive = false {
        didSet {
            guard isActive != oldValue else { return }
            if isActive == false {
                activeTimer.pause()
            } else {
                activeTimer.start()
            }
        }
    }

    private lazy var timer: Repeater = {
        let repeater = Repeater.every(.seconds(interval)) { [unowned self] (repeater) in
            self.performQueryWeather()
        }
        repeater.pause()
        return repeater;
    }()
    
    private lazy var activeTimer: Repeater = {
        let repeater = Repeater.once(after: .seconds(60)) { timer in
            self.isActive = false
        }
        repeater.pause()
        return repeater;
    }()
    
    private func performQueryWeather() {
        guard NEWeatherTask.needPushWeatherInfo, !isActive else { return }
        isActive = true
        DispatchQueue.main.async {
            if NELocationManager.canRequestLocation() {
                NELog.i("requestLocation")
                self.locationManager.startUpdatingLocation()
                return
            }
            self.queryWeatherInfo()
        }
    }

    private func queryWeatherInfo() {
        isActive = true
        guard NEWeatherTask.needPushWeatherInfo,
            let location = NEWeatherTask.latestLocation else { self.isActive = false; return }
        if !NEWeatherInfo.needUpdateQuery {
            toPushWeaterInfo()
            return
        }

        NENetworkingManger.weater(location, NEWeatherTask.queryFailedCount) { [unowned self] (response) in
            switch response.result {
            case .success(let text): NEWeatherInfo.parseAndCache(text)
            case .failure(_): break
            }
            self.toPushWeaterInfo()
        } 
        
    }
    
    private func toPushWeaterInfo() {
        guard NEWeatherInfo.hourItems != nil else { isActive = false;  return  }
        NEWeatherTask.updatePushSubject.onNext(NEWeatherInfo.shared)
    }
}


// MARK: -  queryFailedCount

extension NEWeatherTask {

    private static var queryFailedCount: Int {
        get {
            if _queryFailedCount != nil { return _queryFailedCount! }
            _queryFailedCount = UserDefaults.standard.integer(forKey: kQueryFailedCountKey)
            return _queryFailedCount!
        } set {
            _queryFailedCount = newValue
        }
    }
    
    private static let kQueryFailedCountKey: String = "kQueryFailedCountKey"

    private static var _queryFailedCount: Int? {
        didSet {
            guard _queryFailedCount != oldValue else { return }
            UserDefaults.standard.set(_queryFailedCount, forKey: kQueryFailedCountKey)
            UserDefaults.standard.synchronize()
        }
    }
}

// MARK: -  LatestLocation

extension NEWeatherTask {

    static let kLatestLocationKey: String = "kWeatherLatestLocationKey"
    static var latestLocation: CLLocationCoordinate2D? {
        get {
            if _latestLocation != nil { return _latestLocation! }
            if let cache = UserDefaults.standard.dictionary(forKey: kLatestLocationKey),
                let latitude: Double = cache["latitude"] as? Double,
                let longitude: Double = cache["longitude"] as? Double {
                _latestLocation = CLLocationCoordinate2D(latitude: latitude, longitude: longitude)
            }
            return _latestLocation
        } set {
            _latestLocation = newValue
        }
    }
    
    private static var _latestLocation: CLLocationCoordinate2D? {
        didSet {
            guard _latestLocation != oldValue else { return }
            let dictionary = ["latitude": _latestLocation?.latitude,"longitude": _latestLocation?.longitude]
            UserDefaults.standard.set(dictionary, forKey: kLatestLocationKey)
            UserDefaults.standard.synchronize()
        }
    }
    
    ///: - 天气位置：地理反编码
    private static let geocoder = CLGeocoder()
    private static func _reverseGeocodecityName(_ cmpt: @escaping (()->Void)) {
        
        if country_TW {
            cityName = "";
            cmpt();
            return
        }
        
        guard let lastLocation = latestLocation else { cmpt(); return }
        let location = CLLocation(latitude: lastLocation.latitude, longitude: lastLocation.longitude)
        geocoder.reverseGeocodeLocation(location) { (placemarks, error) in
            if let placemark = placemarks?.first, let locality = placemark.locality {
                cityName = locality
            }
            NELog.i("cityName: \(cityName)")
            cmpt()
        }
    }
    
    /// : - 台湾区域
    /// 用于天气城市名隐藏（台湾二货不认为是中国的台湾省）
    static var country_TW: Bool {
        let locale = NSLocale(localeIdentifier: Locale.current.identifier)
        guard let code:String = locale.object(forKey: .countryCode) as? String else { return false }
        
        let items = ["TW"]
        if items.contains(code) {
            return true
        }
        return false
    }
}



// MARK: -  最后推送的时间戳

extension NEWeatherTask {

    private static var needPushWeatherInfo: Bool {
        (Date().timeIntervalSince1970 - lastPushTime >= 3599)
    }
    
    private static let kLastWeatherPushTimeKey: String = "kLastWeatherPushTimeKey"
    private static var _lastPushTime: TimeInterval? {
        didSet {
            guard _lastPushTime != oldValue else { return }
            UserDefaults.standard.set(_lastPushTime, forKey: kLastWeatherPushTimeKey)
            UserDefaults.standard.synchronize()
        }
    }
}


/// 更新监听
import RxSwift
extension NEWeatherTask {
    private static let disposeBag = DisposeBag()
    private static let updatePushSubject = PublishSubject<NEWeatherInfo>()
    
    /// 更新数据
    public static var updatedPushOutput: Observable<NEWeatherInfo> {
        return updatePushSubject.asObservable()
    }
}


// MARK: -  Public
extension NEWeatherTask {
    
    public static let shared = NEWeatherTask()
    
    ///: - 位置编码城市名
    public static var cityName: String = ""
    
    ///: - 上一次推送天气时：时间戳
    public static var lastPushTime: TimeInterval {
        get {
            if _lastPushTime != nil { return _lastPushTime! }
            _lastPushTime = UserDefaults.standard.double(forKey: kLastWeatherPushTimeKey)
            return _lastPushTime!
        } set {
            if newValue == 0 { isActive = false }
            _lastPushTime = newValue
        }
    }
    
    ///: - 是否活跃
    public static var isActive: Bool {
        get { shared.isActive }
        set { shared.isActive = newValue}
    }

    
    ///: - 启动定时获取更新
    /// 定时间隔（1 小时 ）是否获取更新有更多判断（next、定位位置变化达值）
    public static func start() {
        DispatchQueue.main.async { shared.start() }
    }
    
    ///: - 停止获取更新
    public static func pause() {
        DispatchQueue.main.async { shared.pause() }
    }
    
    ///: - 天气位置：地理反编码
    public static func reverseGeocodecityName(_ cmpt: @escaping (()->Void)) {
        _reverseGeocodecityName(cmpt)
    }
}
