
import Foundation
import CoreBluetooth

let BleLog = Logging(module: "CTBLE", std: .info, file: .none)


// MARK: - Loggable extension

protocol Loggable {
    var logDescription: String { get }
}

extension Data: Loggable {
    var logDescription: String {
        return map { String(format: "%02x", $0) }.joined()
    }
}

extension BluetoothState: Loggable {
    var logDescription: String {
        switch self {
        case .unknown: return "unknown"
        case .resetting: return "resetting"
        case .unsupported: return "unsupported"
        case .unauthorized: return "unauthorized"
        case .poweredOff: return "poweredOff"
        case .poweredOn: return "poweredOn"
        }
    }
}

extension CBCharacteristicWriteType: Loggable {
    var logDescription: String {
        switch self {
        case .withResponse: return "withResponse"
        case .withoutResponse: return "withoutResponse"
        @unknown default:
            return "unknown write type"
        }
    }
}

extension UUID: Loggable {
    var logDescription: String {
        return uuidString
    }
}

extension CBUUID: Loggable {
    @objc var logDescription: String {
        return uuidString
    }
}

extension CBCentralManager: Loggable {
    @objc var logDescription: String {
        return "CentralManager(\(UInt(bitPattern: ObjectIdentifier(self))))"
    }
}

extension CBPeripheral: Loggable {
    @objc var logDescription: String {
        return "Peripheral(uuid: \(uuidIdentifier), name: \(String(describing: name)))"
    }
}

extension CBCharacteristic: Loggable {
    @objc var logDescription: String {
        return "Characteristic(uuid: \(uuid), id: \((UInt(bitPattern: ObjectIdentifier(self)))))"
    }
}

extension CBService: Loggable {
    @objc var logDescription: String {
        return "Service(uuid: \(uuid), id: \((UInt(bitPattern: ObjectIdentifier(self)))))"
    }
}

extension CBDescriptor: Loggable {
    @objc var logDescription: String {
        return "Descriptor(uuid: \(uuid), id: \((UInt(bitPattern: ObjectIdentifier(self)))))"
    }
}

extension CBPeripheralManager: Loggable {
    @objc var logDescription: String {
        return "PeripheralManager(\(UInt(bitPattern: ObjectIdentifier(self))))"
    }
}

extension CBATTRequest: Loggable {
    @objc var logDescription: String {
        return "ATTRequest(\(UInt(bitPattern: ObjectIdentifier(self)))"
    }
}

extension CBCentral: Loggable {
    @objc var logDescription: String {
        return "CBCentral(uuid: \(uuidIdentifier))"
    }
}

@available(iOS 11.0, macOS 10.13, tvOS 11.0, watchOS 4.0, *)
extension CBL2CAPChannel: Loggable {
    @objc var logDescription: String {
        return "CBL2CAPChannel(\(UInt(bitPattern: ObjectIdentifier(self)))"
    }
}

extension Array where Element: Loggable {
    var logDescription: String {
        return "[\(map { $0.logDescription }.joined(separator: ", "))]"
    }
}

