import Foundation

class WeakBox<T: AnyObject> {
    weak var value: T?
    init() {}
    init(value: T) {
        self.value = value
    }
}

extension WeakBox: CustomDebugStringConvertible {
    var debugDescription: String {
        return "WeakBox(\(String(describing: value)))"
    }
}

/// `ThreadSafeBox` 是一个帮助类，它允许以线程安全的方式使用资源(值)。
/// 所有的读和写调用都封装在并发的“DispatchQueue”中，它保护对资源的写操作不超过一个线程
class ThreadSafeBox<T> {
    private let queue = DispatchQueue(label: "com.jpaxh.BluetoothKit.ThreadSafeBox")
    private  var value: T
    init(value: T) {
        self.value = value
    }

    func read<Result: Any>(_ block: (T) -> Result) -> Result {
        var result: Result! = nil
        queue.sync {
            result = block(value)
        }
        return result
    }

    func write(_ block: @escaping (inout T) -> Void) {
        queue.async {
            block(&self.value)
        }
    }

    func writeSync(_ block: @escaping (inout T) -> Void) {
        queue.sync {
            block(&self.value)
        }
    }

    @discardableResult func compareAndSet(compare: (T) -> Bool, set: @escaping (inout T) -> Void) -> Bool {
        var result: Bool = false
        queue.sync {
            result = compare(value)
            if result {
                set(&self.value)
            }
        }
        return result
    }
}

extension ThreadSafeBox: CustomDebugStringConvertible {
    var debugDescription: String {
        return "ThreadSafeBox(\(String(describing: value)))"
    }
}
