/// 日志机制的日志级别.
public enum LoggerLevel: UInt8 {
    /// Logger is disabled
    case none = 255
    /// All logs are monitored.
    case verbose = 0
    /// Only debug logs and of higher importance are logged.
    case debug = 1
    /// Only info logs and of higher importance are logged.
    case info = 2
    /// Only warning logs and of higher importance are logged.
    case warning = 3
    /// Only error logs and of higher importance are logged.
    case error = 4
}


/// 扩展 Comparable
extension LoggerLevel: Comparable {
    
    public static func < (lhs: LoggerLevel, rhs: LoggerLevel) -> Bool {
        return lhs.rawValue < rhs.rawValue
    }

    public static func <= (lhs: LoggerLevel, rhs: LoggerLevel) -> Bool {
        return lhs.rawValue <= rhs.rawValue
    }

    public static func > (lhs: LoggerLevel, rhs: LoggerLevel) -> Bool {
        return lhs.rawValue > rhs.rawValue
    }

    public static func >= (lhs: LoggerLevel, rhs: LoggerLevel) -> Bool {
        return lhs.rawValue >= rhs.rawValue
    }

    public static func == (lhs: LoggerLevel, rhs: LoggerLevel) -> Bool {
        return lhs.rawValue == rhs.rawValue
    }
}


/// 调试日志记录器
class Logger {
    
    private var currentLoggerLevel: LoggerLevel = .verbose

    private var moduleName: String = "SHIRAJO";
    
    private let dformatter: DateFormatter = {
        let dformatter = DateFormatter()
        dformatter.dateFormat = "yyyy-MM-dd HH:mm:ss.SSS"
        return dformatter
    }()
    
    required init(_ name: String = "SRAJO", _ logLevel: LoggerLevel = .verbose) {
          moduleName = name;
          currentLoggerLevel = logLevel
    }

    private func header(with logLevel: LoggerLevel) -> (date:String, text:String) {
        let prefix: String

        switch logLevel {
        case .none:
            prefix = "[\(moduleName)|NONE|"
        case .verbose:
            prefix = "[\(moduleName)|VERB|"
        case .debug:
            prefix = "[\(moduleName)|DEBG|"
        case .info:
            prefix = "[\(moduleName)|INFO|"
        case .warning:
            prefix = "[\(moduleName)|WARN|"
        case .error:
            prefix = "[\(moduleName)|ERRO|"
        }
      
        let now = Date()
        let datetimeStr = dformatter.string(from: now)
        let dateStr = datetimeStr .prefix(10)
        let timeStr = datetimeStr.suffix(12)
        let text = prefix + timeStr + "] "

        return ("\(dateStr)", text)
        
    }
    
    private func contentStart(with logLevel: LoggerLevel) -> String {
        let prefix: String
        
        switch logLevel {
        case .none:
            prefix = "[🖤🖤]: "
        case .verbose:
            prefix = "[💚💚]: "
        case .debug:
            prefix = "[💙💙]: "
        case .info:
            prefix = "[💜💜]: "
        case .warning:
            prefix = "[💛💛]: "
        case .error:
            prefix = "[❤️❤️]: "
        }
        return prefix
    }
    
    /// 记录给定的消息(使用常规字符串参数).
    ///
    /// - Parameters message: 使用闭包提供消息文本. 避免在消息不被记录时执行插值表达式。
    /// - Parameters level: 日志级别.
    /// - Parameters file: 创建消息的文件的文件名.
    /// - Parameters function: 创建消息的函数名.
    /// - Parameters line: 创建消息的文件中的行号.
    private func log(
        _ message: @autoclosure () -> String,
        level: LoggerLevel,
        file: String,
        function: String,
        line: UInt
    ) -> (date:String, text:String) {
        if currentLoggerLevel <= level {
            let head = header(with: level)
            let path = "[\((file as NSString).lastPathComponent):\(line)|\(function)]"
            let headLine = head.text + path + "\n"
            let text = headLine + contentStart(with: level) + message() + "\n"
            return (head.date, text)
        }
        return ("","")
    }
    
    // MARK:  - internal 方法
    

    /// 设置模块名
    /// - Parameter logLevel: 新的日志级别.
    internal func setModelName(_ name: String) {
        self.moduleName = name
    }

    /// 获取当前模块名
    /// - Returns: 当前使用的模块名.
    internal func getModelName() -> String {
        return moduleName
    }
    
    /// 设置日志级别.
    /// - Parameter logLevel: 新的日志级别.
    internal func setLoggerLevel(_ logLevel: LoggerLevel) {
        self.currentLoggerLevel = logLevel
    }

    /// 获取当前日志级别
    /// - Returns: 当前使用的日志级别.
    internal func getLoggerLevel() -> LoggerLevel {
        return currentLoggerLevel
    }
    
    
    /// 记录给定的消息(使用静态字符串参数).
    ///
    /// - Parameters message: 使用闭包提供消息文本. 避免在消息不被记录时执行插值表达式。
    /// - Parameters level: 日志级别.
    /// - Parameters file: 创建消息的文件的文件名.
    /// - Parameters function: 创建消息的函数名.
    /// - Parameters line: 创建消息的文件中的行号.
   internal func log(
        _ message: @autoclosure () -> String,
        level: LoggerLevel,
        file: StaticString,
        function: StaticString,
        line: UInt
    ) -> (date:String, text:String) {
        return log(
            message(),
            level: level,
            file: String(describing: file),
            function: String(describing: function),
            line: line
        )
    }
}
