import Foundation
import RxSwift

/// 日志基础类， 支持Rx的Log 订阅
public class Logging: ReactiveCompatible {
    
    let subject = PublishSubject<(date:String, text:String)>()
    
    let disposeBag = DisposeBag()
        
    let logger:Logger
    
    let file_lvl: LoggerLevel
    
    public required init(module name: String = "SRAJO", std level1: LoggerLevel = .none, file level2: LoggerLevel = .debug) {
        logger = Logger(name, level1)
        file_lvl = level2
        outputFileSubscribe()
    }
       
    /// 设置日志级别.
    /// - Parameter logLevel: 新的日志级别.
    public func setLogLevel(_ logLevel: LoggerLevel) {
        logger.setLoggerLevel(logLevel)
    }
          
    /// 获取当前日志级别
    /// - Returns: 当前使用的日志级别.
    public func getLogLevel() -> LoggerLevel {
        return logger.getLoggerLevel()
    }
    
    
    public func v(
        _ message: @autoclosure () -> String,
        file: StaticString = #file,
        function: StaticString = #function,
        line: UInt = #line
    ) {
        log(with: .verbose, message: message(), file: file, function: function, line: line)
    }

    public func d(
         _ message: @autoclosure () -> String,
         file: StaticString = #file,
         function: StaticString = #function,
         line: UInt = #line
     ) {
         log(with: .debug, message: message(), file: file, function: function, line: line)
     }


    public func i(
        _ message: @autoclosure () -> String,
        file: StaticString = #file,
        function: StaticString = #function,
        line: UInt = #line
    ) {
        log(with: .info, message: message(), file: file, function: function, line: line)
    }

    public func w(
        _ message: @autoclosure () -> String,
        file: StaticString = #file,
        function: StaticString = #function,
        line: UInt = #line
    ) {
        log(with: .warning, message: message(), file: file, function: function, line: line)
    }

    public func e(
        _ message: @autoclosure () -> String,
        file: StaticString = #file,
        function: StaticString = #function,
        line: UInt = #line
    ) {
        log(with: .error, message: message(), file: file, function: function, line: line)
    }
    
    
    func log(
        with logLevel: LoggerLevel,
        message: @autoclosure () -> String,
        file: StaticString,
        function: StaticString,
        line: UInt
    ) {
        let loggedMessage = message()
        let tuples = logger.log(
            loggedMessage,
            level: logLevel,
            file: file,
            function: function,
            line: line
        )
        
        if logger.getLoggerLevel() <= logLevel {
            #if DEBUG
            print(tuples.text)
            #endif
            
            
        }
        if  file_lvl <=  logLevel { subject.onNext(tuples) }
    }
}



// MARK: - Rx 扩展日志器
extension Reactive where Base == Logging {
    /**
     * 这是连续值，它在将日志打印到标准输出之后发出.
     *
     * - 它永远不会失败
     * - 它在 `MainScheduler.instance`
     * - `share(scope: .whileConnected)` 共享策略
     */
    public var log: Observable<(date:String, text:String)> {
        return base.subject.asObserver()
            .observeOn(MainScheduler.instance)
            .catchErrorJustReturn(("",""))
            .share(scope: .whileConnected)
    }
}


// MARK: - 将日记写入文件 Logger 目录下
extension Logging {
    
    enum LoggingOutputError:Error{
        case ErrorNoContent
        case ErrorNoDirectory
        case ErrorNoFile
        case ErrorFile
    }
    
    
    func outputFileSubscribe() {
        subject.subscribe(onNext: {[weak self] (tuples) in
            self?.write(with:tuples.text, to:tuples.date) // 日期作为文件名
        }).disposed(by: disposeBag)
    }
}

extension Logging {
    
    private func write<T>(with message:T, to fileName:String) {
        let content = "\(message)"
        appendText(directory: Logging.directoryURL, fileName: fileName, content: content)
    }
     
    //在文件末尾追加新内容
    private func appendText(directory url: URL, fileName name: String ,content: String) {
        do {
            try ensureValid(url, name, content)
        } catch let error as NSError {
            print("failed to append: \(error)")
        }
    }
    
    private static var fileHandle: FileHandle? = nil

    private func ensureValid(_ url: URL, _ name: String, _ content: String) throws {
        
        if  content.count == 0{
            throw LoggingOutputError.ErrorNoContent
        }
        
        var directory: ObjCBool = ObjCBool(false)
        let exists = FileManager.default.fileExists(atPath: url.path, isDirectory: &directory)
        if exists && directory.boolValue{
        } else {
            do {
                try FileManager.default.createDirectory(atPath: url.path, withIntermediateDirectories: true, attributes: nil)
            } catch {
                throw LoggingOutputError.ErrorNoDirectory
            }
        }
        
        var checkOverdue = false;
        let fileURL = url.appendingPathComponent("\(name).txt")
        if !FileManager.default.fileExists(atPath: fileURL.path) {
            if !FileManager.default.createFile(atPath: fileURL.path, contents: nil) {
                throw LoggingOutputError.ErrorNoFile
            } else {
                checkOverdue = true;
                Logging.fileHandle?.closeFile()
                Logging.fileHandle = nil
            }
        }
        
        do {
            let stringToWrite = content + "\n"
            let data = stringToWrite.data(using: String.Encoding.utf8)!
            if Logging.fileHandle == nil { Logging.fileHandle = try FileHandle(forWritingTo: fileURL) }
            Logging.fileHandle?.seekToEndOfFile()
            Logging.fileHandle?.synchronizeFile()
            Logging.fileHandle?.write(data)
        } catch {
            throw LoggingOutputError.ErrorFile
        }
        
        guard checkOverdue else {return};
        do {
            let items = try FileManager.default.contentsOfDirectory(atPath: url.path)
            try items.sorted { $0 < $1 }.dropLast(5).forEach { (item) in
                let itemURL = url.appendingPathComponent(item)
                try FileManager.default.removeItem(at: itemURL)
            }            
        } catch {
            
        }
    }
}

extension Logging {
    
    private static var directoryURL: URL = {
        let documentURL = FileManager.default.urls(for: .documentDirectory, in: .userDomainMask)[0]
        let directoryURL = documentURL.appendingPathComponent("Logger", isDirectory: true)
        return directoryURL
    } ()
    
    public static func clearFiles() {
        do {
            let items = try FileManager.default.contentsOfDirectory(atPath: directoryURL.path)
            try items.forEach { (item) in
                let itemURL = directoryURL.appendingPathComponent(item)
                try FileManager.default.removeItem(at: itemURL)
            }
        } catch {
            
        }
    }
}
