import Foundation
import CoreBluetooth

/// 方便类，帮助阅读广告数据
public struct AdvertisementData {
    /// 包含外围设备所有广告信息的广告数据字典。
    public let advertisementData: [String: Any]

    /// 基于CoreBluetooth的字典创建广告数据
    /// - parameter advertisementData: Core Bluetooth 的广告数据
    public init(advertisementData: [String: Any]) {
        self.advertisementData = advertisementData
    }

    /// 包含外设的本地名称的字符串.
    public var localName: String? {
        return advertisementData[CBAdvertisementDataLocalNameKey] as? String
    }

    /// 外围设备制造商数据的数据对象.
    public var manufacturerData: Data? {
        return advertisementData[CBAdvertisementDataManufacturerDataKey] as? Data
    }

    /// 包含特定服务的分发数据.
    /// 键是CBUUID对象：表示CBService uuid,
    /// 值是Data对象：代表特定于服务的数据
    public var serviceData: [CBUUID: Data]? {
        return advertisementData[CBAdvertisementDataServiceDataKey] as? [CBUUID: Data]
    }

    /// 需要公布的服务的`UUID`数组
    public var serviceUUIDs: [CBUUID]? {
        return advertisementData[CBAdvertisementDataServiceUUIDsKey] as? [CBUUID]
    }

    /// CBUUID对象的数组，表示在“overflow”中找到的CBService uuid
    /// 区域的广告数据, 如果设备资源不足这些属性可能不会被公布
    public var overflowServiceUUIDs: [CBUUID]? {
        return advertisementData[CBAdvertisementDataOverflowServiceUUIDsKey] as? [CBUUID]
    }

    /// 包含外设的传输功率值.
    /// 如果广播(外围设备)可用此键和值
    /// 在其广播包中提供其Tx功率级别.
    /// 利用RSSI值和Tx功率级，可以计算出路径损耗.
    public var txPowerLevel: NSNumber? {
        return advertisementData[CBAdvertisementDataTxPowerLevelKey] as? NSNumber
    }

    /// 指示广告事件类型是否可连接.
    /// 这个键的值是一个NSNumber对象。您可以使用此值来确定是否
    /// 外设在特定时刻是可连接的.
    public var isConnectable: Bool? {
        return advertisementData[CBAdvertisementDataIsConnectable] as? Bool
    }

    /// CBUUID对象的数组，表示CBService uuid.
    public var solicitedServiceUUIDs: [CBUUID]? {
        return advertisementData[CBAdvertisementDataSolicitedServiceUUIDsKey] as? [CBUUID]
    }
}
