import Foundation
import RxSwift
import CoreBluetooth


/// 接收“CentralManagerRestoredState”作为参数的闭包
public typealias OnWillRestoreCentralManagerState = (CentralManagerRestoredState) -> Void

extension CentralManager {

    // MARK: 状态恢复


    /// Creates new `CentralManager` instance, 支持蓝牙状态恢复.
    /// - warning: 如果将后台队列传递给方法，请确保在主线程上观察与UI相关的代码的结果。
    /// - parameter queue: 接收蓝牙回调的队列。默认情况下使用主线程，所有操作和事件都在主线程上执行和接收。
    /// - parameter options: 一个可选的字典，包含中心管理器的初始化选项。有关它的更多信息，请参阅 [Central Manager initialization options](https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBCentralManager_Class/index.html)
    /// - parameter onWillRestoreCentralManagerState: 已恢复状态时调用闭包.
    ///
    /// - seealso: `OnWillRestoreCentralManagerState`
    public convenience init(queue: DispatchQueue = .main,
                            options: [String: Any]? = nil,
                            onWillRestoreCentralManagerState: OnWillRestoreCentralManagerState? = nil) {
        self.init(queue: queue, options: options)
        if let onWillRestoreCentralManagerState = onWillRestoreCentralManagerState {
            listenOnWillRestoreState(onWillRestoreCentralManagerState)
        }
    }

    /// Creates new `CentralManager`
    /// - parameter centralManager: 中心实例，用于执行所有必要的操作
    /// - parameter delegateWrapper: CoreBluetooth中央管理器回调上的包装器。
    /// - parameter peripheralProvider: 提供外围设备和外围包装器的提供商
    /// - parameter connector: 连接器实例，用于建立与外围设备的连接
    /// - parameter onWillRestoreState: 已恢复状态时调用闭包.
    convenience init(
        centralManager: CBCentralManager,
        delegateWrapper: CBCentralManagerDelegateWrapper,
        peripheralProvider: PeripheralProvider,
        connector: Connector,
        onWillRestoreCentralManagerState: @escaping OnWillRestoreCentralManagerState
        ) {
        self.init(
            centralManager: centralManager,
            delegateWrapper: delegateWrapper,
            peripheralProvider: peripheralProvider,
            connector: connector
        )
        listenOnWillRestoreState(onWillRestoreCentralManagerState)
    }

    /// 发出 `RestoredState` 实例, 当‘CentralManager’状态恢复后,
    /// Should only be called once in the lifetime of the app
    /// - returns: Observable which emits next events state has been restored
    func listenOnWillRestoreState(_ handler: @escaping OnWillRestoreCentralManagerState) {
        _ = restoreStateObservable
            .subscribe(onNext: { handler($0) })
    }

    var restoreStateObservable: Observable<CentralManagerRestoredState> {
        return delegateWrapper.willRestoreState
            .take(1)
            .flatMap { [weak self] dict -> Observable<CentralManagerRestoredState> in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                return .just(CentralManagerRestoredState(restoredStateDictionary: dict, centralManager: strongSelf))
            }
    }
}
