import Foundation
import RxSwift
import CoreBluetooth

/// 发生设备断开事件时接收到的错误
public typealias DisconnectionReason = Error

/// CentralManager 是一个实现 ReactiveX API 的类，它封装了所有核心蓝牙管理器的功能，允许发现、连接到远程外围设备等。
/// 您可以通过发现附近外围设备的可用服务来开始使用这个类。在调用任何公共 ‘CentralManager’ 的功能之前，您应该确保蓝牙处于打开状态。
/// 它可以通过调用和观察 ‘observeState()’ 的返回值来实现，然后用 ‘scanForPeripherals(_:options:)’ 来查找它:
/// ```
/// let disposable = centralManager.observeState
///     .startWith(centralManager.state)
///     .filter { $0 == .poweredOn }
///     .take(1)
///     .flatMap { centralManager.scanForPeripherals(nil) }
/// ```
/// 结果你将收到 ‘ScannedPeripheral’，它包含了外设对象、广告数据和外设在发现过程中注册的RSSI.
/// 然后可以 'establishConnection(_:options:) '(建立连接) 并执行其他操作.
/// 你也可以简单地停止扫描，只要处理它:
/// ```
/// disposable.dispose()
/// ```
public class CentralManager: ManagerType {

    /// 实现CBCentralManager
    public let manager: CBCentralManager


    let peripheralProvider: PeripheralProvider

    let delegateWrapper: CBCentralManagerDelegateWrapper

    /// 在访问任何内部结构之前应该使用的锁
    private let lock = NSLock()

    /// 正在进行扫描 disposable
    private var scanDisposable: Disposable?

    /// 连接器实例: 用于建立与外围设备的连接
    private let connector: Connector

    // MARK: Initialize

    /// 创建 `CentralManager`
    /// - parameter centralManager: 中心实例，用于执行所有必要的操作
    /// - parameter delegateWrapper: 'CoreBluetooth CentralManager' 回调上的包装器
    /// - parameter peripheralProvider: 提供外围设备和外围包装器的提供商
    /// - parameter connector: 连接器实例，用于建立与外围设备的连接
    init(
        centralManager: CBCentralManager,
        delegateWrapper: CBCentralManagerDelegateWrapper,
        peripheralProvider: PeripheralProvider,
        connector: Connector
    ) {
        self.manager = centralManager
        self.delegateWrapper = delegateWrapper
        self.peripheralProvider = peripheralProvider
        self.connector = connector
        centralManager.delegate = delegateWrapper
    }

    /// 创建新的' CentralManager '实例。默认情况下，所有操作和事件都在主线程上执行和接收.
    /// - warning: 如果将后台队列传递给方法，请确保在主线程上观察与UI相关的代码的结果.
    /// - parameter queue: 接收蓝牙回调的队列，默认情况下使用主线程.
    /// - parameter options: 一个可选的字典，包含中心管理器的初始化选项.
    /// 有关它的更多信息，请参考[中央管理器初始化选项]
    /// (https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBCentralManager_Class/index.html)
    public convenience init(queue: DispatchQueue = .main,
                            options: [String: Any]? = nil) {
        let delegateWrapper = CBCentralManagerDelegateWrapper()
        let centralManager = CBCentralManager(delegate: delegateWrapper, queue: queue, options: options)
        self.init(
            centralManager: centralManager,
            delegateWrapper: delegateWrapper,
            peripheralProvider: PeripheralProvider(),
            connector: Connector(centralManager: centralManager, delegateWrapper: delegateWrapper)
        )
    }

    /// 将 'BluetoothKit' 蓝牙工具包委托('delegateWrapper')连接到'CBCentralManager'.
    /// 当CBCentralManager的委托被重新分配到外部时，此方法非常有用.
    /// 'BluetoothKit' 工具包库(例如'，'CBCentralManager'用于其他库或以非响应方式使用)
    public func attach() {
        manager.delegate = delegateWrapper
    }

    // MARK: State

    public var state: BluetoothState {
        return BluetoothState(rawValue: manager.state.rawValue) ?? .unsupported
    }

    public func observeState() -> Observable<BluetoothState> {
        return self.delegateWrapper.didUpdateState.asObservable()
    }

    // MARK: Scanning
    
    /// 订阅返回的可观察对象后扫描外围设备。
    /// 第一个参数的'serviceUUIDs'是一个服务UUID数组，需要通过一个外设来实现。
    /// 如果用户不想过滤任何外设，可以使用nil。此外，还可以传递 'CBCentralManager specific options' 以允许进一步定制。
    /// (https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBCentralManager_Class/index.html)
    ///
    /// 默认情况下，扫描是无限流，需要用户停止。例如，这可以通过限制扫描一定数量的外围设备或时间:
    /// ```
    /// centralManager.scanForPeripherals(withServices: nil)
    ///     .timeout(3.0, timeoutScheduler)
    ///     .take(2)
    /// ```
    ///
    /// 只能有一个正在进行的扫描. 如果在已经在进行扫描时调用此方法，则产生 'BluetoothError.scanInProgress' 错误。
    /// 结果你将收到'ScannedPeripheral'，它包含了外设对象、广告数据和外设在发现过程中注册的RSSI。
    /// 然后可以'establishConnection(_:options:)'并执行其他操作。
    ///
    /// - parameter serviceUUIDs: 要搜索的外设服务CBUUID数组，空值将接受所有外围设备。
    /// - parameter options: 可选的扫描选项.
    /// - returns: 无限的扫描外围设备流.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.scanInProgress`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func scanForPeripherals(withServices serviceUUIDs: [CBUUID]?, options: [String: Any]? = nil)
                    -> Observable<ScannedPeripheral> {
        let observable: Observable<ScannedPeripheral> = Observable.create { [weak self] observer in
            guard let strongSelf = self else {
                observer.onError(BluetoothError.destroyed)
                return Disposables.create()
            }
            strongSelf.lock.lock(); defer { strongSelf.lock.unlock() } /// defer block 里的代码会在函数 return 之前执行
            if strongSelf.scanDisposable != nil {
                observer.onError(BluetoothError.scanInProgress)
                return Disposables.create()
            }
            strongSelf.scanDisposable = strongSelf.delegateWrapper.didDiscoverPeripheral
                    .flatMap { [weak self] (cbPeripheral, advertisment, rssi) -> Observable<ScannedPeripheral> in
                        guard let strongSelf = self else {
                            throw BluetoothError.destroyed
                        }
                        let peripheral = strongSelf.retrievePeripheral(for: cbPeripheral)
                        let advertismentData = AdvertisementData(advertisementData: advertisment)
                        return .just(ScannedPeripheral(peripheral: peripheral,
                                advertisementData: advertismentData, rssi: rssi))
                    }
                    .subscribe(observer)

            strongSelf.manager.scanForPeripherals(withServices: serviceUUIDs, options: options)

            return Disposables.create { [weak self] in
                guard let strongSelf = self else { return }
                // 销毁时去停止并释放扫描
                if strongSelf.state == .poweredOn {
                    strongSelf.manager.stopScan()
                }
                do { strongSelf.lock.lock(); defer { strongSelf.lock.unlock() }
                    strongSelf.scanDisposable?.dispose()
                    strongSelf.scanDisposable = nil
                }
            }
        }

        return ensure(.poweredOn, observable: observable)
    }

    // MARK: 外围的连接管理

    /// 与给定的“Peripheral”建立连接。.
    /// 当连接成功，它发送事件与'Peripheral'。 从这时候起，它可以调用所有需要连接的其他方法。
    /// 当结果观察对象被取消订阅时，连接被自动断开.
    /// 另外，当连接被设备或系统中断或失败时，被观察对象将被取消订阅并发送错误：
    /// ```
    ///  BluetoothError.peripheralConnectionFailed
    ///  BluetoothError.peripheralDisconnected
    ///  ```
    /// 此外，您可以通过可选项定制连接的行为： (https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBCentralManager_Class/#//apple_ref/doc/constant_group/Peripheral_Connection_Options)
    ///
    /// - parameter peripheral: ' CentralManager '试图建立连接的'外设'.
    /// - parameter options: 字典来定制连接的行为.
    /// - returns: `Observable` 连接建立后，哪个发出下一个事件.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.peripheralIsAlreadyObservingConnection`
    /// * `BluetoothError.peripheralConnectionFailed`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func establishConnection(_ peripheral: Peripheral, options: [String: Any]? = nil) -> Observable<Peripheral> {
        let observable = connector.establishConnection(with: peripheral, options: options)
        return ensure(.poweredOn, observable: observable)
    }

    // MARK: 检索外设列表

    /// 返回当前连接到“CentralManager”的“外设”列表，其中包含所有指定的“服务”uuid.
    ///
    /// - parameter serviceUUIDs: 服务uuid的列表
    /// - returns: 检索到的外围.  它们处于连接状态，并包含所有在“serviceUUIDs”参数中指定的带有uuid的服务。
    public func retrieveConnectedPeripherals(withServices serviceUUIDs: [CBUUID]) -> [Peripheral] {
        return manager.retrieveConnectedPeripherals(withServices: serviceUUIDs)
            .map { self.retrievePeripheral(for: $0) }
    }

    /// 返回'外设'的列表，其标识符是已知的' CentralManager '
    ///
    /// - parameter identifiers: 应检索的“外围设备”标识符列表.
    /// - returns: 检索到的外围设备.
    public func retrievePeripherals(withIdentifiers identifiers: [UUID]) -> [Peripheral] {
        return manager.retrievePeripherals(withIdentifiers: identifiers)
            .map { self.retrievePeripheral(for: $0) }
    }

    // MARK: 连接和断开观察

    /// 当它被连接时发出'Peripheral'实例
    ///
    /// - parameter peripheral: 可选的“Peripheral”，用于观察连接。当未指定时，它将观察任何“Peripheral”.
    /// - returns: 当‘外围’连接时，发出下一个事件的可观察对象.
    ///
    /// 它是**infinite**流，所以' .complete '不会被调用..
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeConnect(for peripheral: Peripheral? = nil) -> Observable<Peripheral> {
        let observable = delegateWrapper.didConnectPeripheral
            .filter { peripheral != nil ? ($0 == peripheral!.peripheral) : true }
            .map { [weak self] (cbPeripheral: CBPeripheral) -> Peripheral in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                return peripheral ?? strongSelf.retrievePeripheral(for: cbPeripheral)
            }
      return ensure(.poweredOn, observable: observable)
    }

    /// 当它断开连接时发出'Peripheral'实例.
    /// - parameter peripheral: 可选的“Peripheral”，用于观察连接。当未指定时，它将观察任何“Peripheral”.
    /// - returns: 它提供了可选的错误，如果不是“cancelPeripheralConnection”调用，则可能包含有关断开连接原因的更多信息.
    ///
    /// 它是**infinite**流，所以' .complete '不会被调用..
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeDisconnect(for peripheral: Peripheral? = nil) -> Observable<(Peripheral, DisconnectionReason?)> {
        let observable = delegateWrapper.didDisconnectPeripheral
            .filter { peripheral != nil ? ($0.0 == peripheral!.peripheral) : true }
            .map { [weak self] (cbPeripheral, error) -> (Peripheral, DisconnectionReason?) in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                let peripheral = peripheral ?? strongSelf.retrievePeripheral(for: cbPeripheral)
                return (peripheral, error)
            }
        return ensure(.poweredOn, observable: observable)
                .catchError { error in
                    if error is BluetoothError, let peripheral = peripheral {
                        return .concat(.just((peripheral, error)), .error(error))
                    } else {
                        return .error(error)
                    }
                }
    }

    // MARK: - 内部函数

    /// 确保在订阅期间连接指定的“Peripheral”.
    /// - parameter peripheral: 'Peripheral' 应在订阅期间连接.
    /// - returns: “Peripheral”在订阅期间断开连接时, 可观察到它发出错误.
    func ensurePeripheralIsConnected<T>(_ peripheral: Peripheral) -> Observable<T> {
        return .deferred {
            if !peripheral.isConnected {
                throw BluetoothError.peripheralDisconnected(peripheral, nil)
            }
            return self.delegateWrapper.didDisconnectPeripheral
                .filter { $0.0 == peripheral.peripheral }
                .map { (_, error) -> T in
                    throw BluetoothError.peripheralDisconnected(peripheral, error)
            }
        }
    }
    
    
    /// CBPeripheral 对象来创建 Peripheral 对象
    /// - Parameter cbPeripheral: CBPeripheral 对象
    func retrievePeripheral(for cbPeripheral: CBPeripheral) -> Peripheral {
        return peripheralProvider.provide(for: cbPeripheral, centralManager: self)
    }
}
