import Foundation
import CoreBluetooth

/// 当 “CentralManagerRestoredState” 将被删除时，它应该被删除
protocol CentralManagerRestoredStateType {
    var restoredStateData: [String: Any] { get }
    var centralManager: CentralManager { get }
    var peripherals: [Peripheral] { get }
    var scanOptions: [String: AnyObject]? { get }
    var services: [Service] { get }
}

/// Convenience class 这有助于读取已恢复的 CentralManager 的状态。
public struct CentralManagerRestoredState: CentralManagerRestoredStateType {

    /// 恢复状态字典数据.
    public let restoredStateData: [String: Any]

    public unowned let centralManager: CentralManager
    /// 基于CoreBluetooth 状态恢复的字典信息数据
    /// - parameter restoredStateDictionary: Core Bluetooth's 恢复状态数据
    /// - parameter centralManager: `CentralManager` 已恢复的状态的实例.
    init(restoredStateDictionary: [String: Any], centralManager: CentralManager) {
        restoredStateData = restoredStateDictionary
        self.centralManager = centralManager
    }

    /// 已恢复的“外围”对象的数组.
    /// 在应用程序被系统终止的时候， 这些是连接到中央管理器的外围设备 (或有一个连接等待)
    public var peripherals: [Peripheral] {
        let objects = restoredStateData[CBCentralManagerRestoredStatePeripheralsKey] as? [AnyObject]
        guard let arrayOfAnyObjects = objects else { return [] }

        #if swift(>=4.1)
        let cbPeripherals = arrayOfAnyObjects.compactMap { $0 as? CBPeripheral }
        #else
        let cbPeripherals = arrayOfAnyObjects.flatMap { $0 as? CBPeripheral }
        #endif

        return cbPeripherals.map { centralManager.retrievePeripheral(for: $0) }
    }

    /// 包含正在使用的所有外围扫描选项的字典
    /// 这些选项在应用程序被系统终止时由中央管理器使用.
    public var scanOptions: [String: AnyObject]? {
        return restoredStateData[CBCentralManagerRestoredStatePeripheralsKey] as? [String: AnyObject]
    }

    /// 已恢复的“服务”对象的数组.
    /// 这些是中央管理器在应用程序被系统终止时扫描的所有服务
    public var services: [Service] {
        let objects = restoredStateData[CBCentralManagerRestoredStateScanServicesKey] as? [AnyObject]
        guard let arrayOfAnyObjects = objects else { return [] }

        #if swift(>=4.1)
        let cbServices = arrayOfAnyObjects.compactMap { $0 as? CBService }
        #else
        let cbServices = arrayOfAnyObjects.flatMap { $0 as? CBService }
        #endif

        return cbServices.map { Service(peripheral: centralManager.retrievePeripheral(for: $0.peripheral!),
                                        service: $0) }
    }
}
