import Foundation
import RxSwift
import CoreBluetooth

/// `Connector` 负责建立与外围设备的连接的类.
class Connector {
    let centralManager: CBCentralManager
    let delegateWrapper: CBCentralManagerDelegateWrapper
    let connectedBox: ThreadSafeBox<Set<UUID>> = ThreadSafeBox(value: [])
    let disconnectingBox: ThreadSafeBox<Set<UUID>> = ThreadSafeBox(value: [])

    init(
        centralManager: CBCentralManager,
        delegateWrapper: CBCentralManagerDelegateWrapper
    ) {
        self.centralManager = centralManager
        self.delegateWrapper = delegateWrapper
    }

    /// 与给定的“外围设备”建立连接.
    /// 有关更多信息，请参见 `CentralManager.establishConnection(with:options:)`
    func establishConnection(with peripheral: Peripheral, options: [String: Any]? = nil)
        -> Observable<Peripheral> {
            return .deferred { [weak self] in
                guard let strongSelf = self else {
                    return Observable.error(BluetoothError.destroyed)
                }
                let connectionObservable = strongSelf.createConnectionObservable(for: peripheral, options: options)

                let waitForDisconnectObservable = strongSelf.delegateWrapper.didDisconnectPeripheral
                    .filter { $0.0 == peripheral.peripheral }
                    .take(1)
                    .do(onNext: { [weak self] _ in
                        guard let strongSelf = self else { return }
                        strongSelf.disconnectingBox.write { $0.remove(peripheral.identifier) }
                    })
                    .map { _ in peripheral }
                let isDisconnectingObservable: Observable<Peripheral> = Observable.create { observer in
                    var isDiconnecting = strongSelf.disconnectingBox.read { $0.contains(peripheral.identifier) }
                    let isDisconnected = peripheral.state == .disconnected
                    // it means that peripheral is already disconnected, but we didn't update disconnecting box
                    if isDiconnecting && isDisconnected {
                        strongSelf.disconnectingBox.write { $0.remove(peripheral.identifier) }
                        isDiconnecting = false
                    }
                    if !isDiconnecting {
                        observer.onNext(peripheral)
                    }
                    return Disposables.create()
                }
                return waitForDisconnectObservable.amb(isDisconnectingObservable)
                    .flatMap { _ in connectionObservable }
            }
    }

    private  func createConnectionObservable(
        for peripheral: Peripheral,
        options: [String: Any]? = nil
    ) -> Observable<Peripheral> {
        return Observable.create { [weak self] observer in
            guard let strongSelf = self else {
                observer.onError(BluetoothError.destroyed)
                return Disposables.create()
            }

            let connectingStarted = strongSelf.connectedBox.compareAndSet(
                compare: { !$0.contains(peripheral.identifier) },
                set: { $0.insert(peripheral.identifier) }
            )

            guard connectingStarted else {
                observer.onError(BluetoothError.peripheralIsAlreadyObservingConnection(peripheral))
                return Disposables.create()
            }

            let connectedObservable = strongSelf.createConnectedObservable(for: peripheral)
            let failToConnectObservable = strongSelf.createFailToConnectObservable(for: peripheral)
            let disconnectedObservable = strongSelf.createDisconnectedObservable(for: peripheral)

            let disposable = connectedObservable.amb(failToConnectObservable)
                .do(onNext: { observer.onNext($0) })
                .flatMap { _ in disconnectedObservable }
                .subscribe(onError: { observer.onError($0) })

            // 苹果建议，在获取外设后，始终连接外设.
            // https://developer.apple.com/library/archive/documentation/NetworkingInternetWeb/Conceptual/CoreBluetooth_concepts/BestPracticesForInteractingWithARemotePeripheralDevice/BestPracticesForInteractingWithARemotePeripheralDevice.html#//apple_ref/doc/uid/TP40013257-CH6-SW9
            //
            // 摘自“重新连接到外围设备”
            //
            // "使用retrieveperipheralswithidentifier:方法检索已知外围设备的列表—您已经发现的或在过去连接到的外围设备。
            // 如果您要查找的外围设备在列表中，请尝试连接到它。"
            //
            // "假设用户找到并选择了所需的外设，那么通过调用CBCentralManager类的connectPeripheral:options:方法，将它本地连接到您的应用程序.
            // (即使设备已经连接到系统，您仍然必须将它本地连接到您的应用程序，以开始探索和与它交互) When the local
            // 当本地连接建立后，中央管理器调用centralManager:didConnectPeripheral:其委托对象的方法，外围设备被成功重新连接."
            
            strongSelf.centralManager.connect(peripheral.peripheral, options: options)

            return Disposables.create { [weak self] in
                guard let strongSelf = self else { return }
                disposable.dispose()
                let isConnected = strongSelf.connectedBox.read { $0.contains(peripheral.identifier) }
                if isConnected {
                    if strongSelf.centralManager.state == .poweredOn {
                        strongSelf.disconnectingBox.write { $0.insert(peripheral.identifier) }
                        strongSelf.centralManager.cancelPeripheralConnection(peripheral.peripheral)
                    }
                    strongSelf.connectedBox.write { $0.remove(peripheral.identifier) }
                }
            }
        }
    }

    private  func createConnectedObservable(for peripheral: Peripheral) -> Observable<Peripheral> {
        return delegateWrapper.didConnectPeripheral
            .filter { $0 == peripheral.peripheral }
            .take(1)
            .map { _ in peripheral }
    }

    private  func createDisconnectedObservable(for peripheral: Peripheral) -> Observable<Peripheral> {
        return delegateWrapper.didDisconnectPeripheral
            .filter { $0.0 == peripheral.peripheral }
            .take(1)
            .do(onNext: { [weak self] _ in
                guard let strongSelf = self else { return }
                strongSelf.connectedBox.write { $0.remove(peripheral.identifier) }
                strongSelf.disconnectingBox.write { $0.remove(peripheral.identifier) }
            })
            .map { (_, error) -> Peripheral in
                throw BluetoothError.peripheralDisconnected(peripheral, error)
            }
    }

    private  func createFailToConnectObservable(for peripheral: Peripheral) -> Observable<Peripheral> {
        return delegateWrapper.didFailToConnectPeripheral
            .filter { $0.0 == peripheral.peripheral }
            .take(1)
            .do(onNext: { [weak self] _ in
                guard let strongSelf = self else { return }
                strongSelf.connectedBox.write { $0.remove(peripheral.identifier) }
            })
            .map { (_, error) -> Peripheral in
                throw BluetoothError.peripheralConnectionFailed(peripheral, error)
            }
    }
}
