import Foundation
import RxSwift
import CoreBluetooth

/// Characteristic 一个实现ReactiveX的类, 它封装了与交互相关的CoreBluetooth功能 [CBCharacteristic](https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBCharacteristic_Class/)
public class Characteristic {
    /// CBCharacteristic 实例
    public let characteristic: CBCharacteristic

    /// 包含 characteristic 的服务
    public let service: Service

    /// characteristic 当前值。如果value不存在 - 它是' nil '.
    public var value: Data? {
        return characteristic.value
    }

    /// “特征”实例的蓝牙UUID.
    public var uuid: CBUUID {
        return characteristic.uuid
    }

    /// 标记该characteristic 是否开启了通知，如果正在通知则为：true
    public var isNotifying: Bool {
        return characteristic.isNotifying
    }

    /// characteristic 的属性. 更多信息请参考：[CBCharacteristicProperties]
    /// (https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBCharacteristic_Class/#//apple_ref/c/tdef/CBCharacteristicProperties)

    public var properties: CBCharacteristicProperties {
        return characteristic.properties
    }

    /// 这个属性的值是一个“Descriptor”对象数组. 它们提供了关于特征值的更详细的信息。
    public var descriptors: [Descriptor]? {
        return characteristic.descriptors?.map { Descriptor(descriptor: $0, characteristic: self) }
    }

    init(characteristic: CBCharacteristic, service: Service) {
        self.characteristic = characteristic
        self.service = service
    }

    convenience init(characteristic: CBCharacteristic, peripheral: Peripheral) {
        let service = Service(peripheral: peripheral, service: characteristic.service!)
        self.init(characteristic: characteristic, service: service)
    }

    /// 触发特征描述符发现的函数.
    /// - returns: 一旦被发现，‘Single’会发出带有‘Descriptor’数组实例的‘next’事件.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorsDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func discoverDescriptors() -> Single<[Descriptor]> {
        return service.peripheral.discoverDescriptors(for: self)
    }

    /// 观察此 characteristic 写操作.
    /// - Returns: ' Observable '会在每次写操作发生时，发出' next '的' Characteristic '实例.
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeWrite() -> Observable<Characteristic> {
        return service.peripheral.observeWrite(for: self)
    }

    /// 当某一特征的isNotyfing值发生变化时， 允许知道确切的时间
    ///
    /// - returns: 当isNoytfing值发生变化时，`Observable` 发送 `Characteristic
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicSetNotifyValueFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeNotifyValue() -> Observable<Characteristic> {
        return service.peripheral.observeNotifyValue(for: self)
    }

    /// 触发将数据写入特征的函数。在订阅“Observable”之后调用Write
    /// 这个函数的行为强烈依赖于 [CBCharacteristicWriteType]
    /// (https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBPeripheral_Class/#//apple_ref/swift/enum/c:@E@CBCharacteristicWriteType), 所以在使用这个方法之前一定要检查一下.
    /// - parameter data: 将要写入到特征中的数据
    /// - parameter type: 写操作类型. 可能的值: `.withResponse`, `.withoutResponse`
    /// - returns: `Single` 它的发射依赖于传递给函数调用的‘CBCharacteristicWriteType’l.
    /// 行为如下:
    ///
    /// - `withResponse` -  `Observable` 发送带有`Characteristic`的`next`事件， 实例写被确认没有任何错误时.
    /// 如果发生了任何问题，就会发出错误.
    /// - `withoutResponse` - `Observable` 发送带有`Characteristic`的`next`事件，实例一次写入被调用.
    /// 此调用的结果未被检查, 所以用户不能确定
    /// 如果一切顺利的话，不会发出错误
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func writeValue(_ data: Data, type: CBCharacteristicWriteType) -> Single<Characteristic> {
        return service.peripheral.writeValue(data, for: self, type: type)
    }

    /// 允许观察“特征”实例的值更新的函数.
    /// - Returns: `Observable`发送带有`Characteristic`的`next`事件，任何时候发生值改变时.
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeValueUpdate() -> Observable<Characteristic> {
        return service.peripheral.observeValueUpdate(for: self)
    }

    /// 该函数发出读取`Characteristic` 实例的当前值.
    /// Read在订阅“Observable”之后调用.
    /// - Returns: `Single` 发送带有`Characteristic`的`next`事件， 当值准备读取时.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func readValue() -> Single<Characteristic> {
        return service.peripheral.readValue(for: self)
    }

    /// 设置特征通知，以便在给定特征被更改时接收回调。
    /// 返回的被观察对象将在每个通知变化上发出“特征”。.
    /// 可以为相同的特征设置更多的可观察对象，并且通知的生命周期将在它们之间共享.
    ///
    /// 一旦这个可观察对象被取消订阅，通知将自动取消注册
    ///
    /// - returns: `Observable` 发送带有`Characteristic`的 `next`事件， 当给定的特征值发生改变时.
    ///
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeValueUpdateAndSetNotification() -> Observable<Characteristic> {
        return service.peripheral.observeValueUpdateAndSetNotification(for: self)
    }
}

extension Characteristic: Equatable {}
extension Characteristic: UUIDIdentifiable {}

/// 比较两个特征。当它们的uuid相同时，特征是相同的.
///
/// - parameter lhs: 第一个特征
/// - parameter rhs: 第二个特征
/// - returns: 如果两个特征相同，则为真.
public func == (lhs: Characteristic, rhs: Characteristic) -> Bool {
    return lhs.characteristic == rhs.characteristic
}
