import Foundation
import CoreBluetooth
import RxSwift

/// Descriptor 是一个实现ReactiveX的类, 它封装了与CoreBluetooth交互相关的功能
/// [CBDescriptor](https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBDescriptor_Class/)
/// Descriptors 提供关于特征值的更多信息
public class Descriptor {

    /// CoreBluetooth descriptor 实例
    public let descriptor: CBDescriptor

    /// 描述符所属的特征.
    public let characteristic: Characteristic

    /// 描述符”实例的蓝牙UUID.
    public var uuid: CBUUID {
        return descriptor.uuid
    }

    /// 描述符的值， 它可以在“描述符”实例上写入和读取.
    public var value: Any? {
        return descriptor.value
    }

    init(descriptor: CBDescriptor, characteristic: Characteristic) {
        self.descriptor = descriptor
        self.characteristic = characteristic
    }

    convenience init(descriptor: CBDescriptor, peripheral: Peripheral) {
        let service = Service(peripheral: peripheral, service: descriptor.characteristic!.service!)
        let characteristic = Characteristic(characteristic: descriptor.characteristic!, service: service)
        self.init(descriptor: descriptor, characteristic: characteristic)
    }

    /// 观察发生在描述符上的写操作.
    /// - Returns: Observable 发送带有`Descriptor`的 `next`事件，当 `Descriptor` 实例发生写操作时.
    /// 它是**infinite**流，所以' .complete '不会被调用。
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeWrite() -> Observable<Descriptor> {
        return characteristic.service.peripheral.observeWrite(for: self)
    }

    /// 函数触发将数据写入描述符。在订阅“Observable”之后调用Write.
    /// - Parameter data: 将要写入`Descriptor` 实例的数据
    /// - Returns: `Single` 发送带Descriptor的`Next， 当写入值成功时.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func writeValue(_ data: Data) -> Single<Descriptor> {
        return characteristic.service.peripheral.writeValue(data, for: self)
    }

    /// 观察“描述符”实例的值更新.
    /// - Returns: 被观察对象，它会在每次值改变时发出带有“描述符”实例的“next”。
    /// 它是**infinite**流，所以' .complete '不会被调用。
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeValueUpdate() -> Observable<Descriptor> {
        return characteristic.service.peripheral.observeValueUpdate(for: self)
    }

    /// 触发读取“描述符”实例的当前值。
    /// Read在订阅“Observable”之后调用.
    /// - Returns: `Single` 发送带有descriptor的 `next` ，当读取描述值时.
    ///
    /// Observable can ends with following errors:
    /// * `BluetoothError.descriptorReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func readValue() -> Single<Descriptor> {
        return characteristic.service.peripheral.readValue(for: self)
    }
}

extension Descriptor: Equatable {}

/// 比较两个描述符。当它们的uuid相同时，描述符是相同的。
///
/// - parameter lhs: 第一个描述
/// - parameter rhs: 第二个描述
/// - returns: 当两个描述相等时放回True.
public func == (lhs: Descriptor, rhs: Descriptor) -> Bool {
    return lhs.descriptor == rhs.descriptor
}
