import Foundation
import RxSwift
import CoreBluetooth

public protocol ManagerType: AnyObject {
    associatedtype Manager

    /// 实现CBManager
    var manager: Manager { get }
    /// ‘CentralManager '实例的当前状态由'蓝牙状态'描述，它相当于
    /// [CBManagerState](https://developer.apple.com/documentation/corebluetooth/cbmanagerstate).
    var state: BluetoothState { get }

    /// “蓝牙状态”描述的“CBManager”实例的连续状态，相当于  [CBManagerState](https://developer.apple.com/documentation/corebluetooth/cbmanagerstate).
    /// - returns: Observable 发送带BluetoothState的 `next` 事件，当状态发生改变时.
    ///
    /// 它是**infinite**流，所以' .complete '不会被调用.
    func observeState() -> Observable<BluetoothState>
}

public extension ManagerType {
    /// 确保“state”是且将是订阅期间“CentralManager”的唯一状态.
    /// 否则会发出错误.
    /// - parameter state: 应该在订阅期间出现的`'蓝牙状态'
    /// - parameter observable: 合并潜在的错误.
    /// - returns: 新观测值，将误差与源观测值合并.
    func ensure<T>(_ state: BluetoothState, observable: Observable<T>) -> Observable<T> {
        return .deferred { [weak self] in
            guard let strongSelf = self else { throw BluetoothError.destroyed }
            let statesObservable = strongSelf.observeState()
                .startWith(strongSelf.state)
                .filter { $0 != state && BluetoothError(state: $0) != nil }
                .map { state -> T in throw BluetoothError(state: state)! }
            return .absorb(statesObservable, observable)
        }
    }
}
