import CoreBluetooth

/// ServiceIdentifier协议指定关于设备的服务的信息应该是什么样的.
/// 示例实现可能如下所示:
///
/// ```
/// enum DeviceService: String, ServiceIdentifier {
///     case deviceInformation = "180A"
///     var uuid: CBUUID {
///         return CBUUID(string: self.rawValue)
///     }
/// }
/// ```
/// 在实现了这个和其他协议的CharacteristicIdentifier之后，您可以更容易地调用方法
/// 多亏了library里方便的方法. 这些方法完成了所有繁重的工作，比如发现服务、获取服务
/// 方便你用于缓存它。建议您使用它
public protocol ServiceIdentifier {
    /// 服务的唯一标识符.
    var uuid: CBUUID { get }
}

/// 特征协议指定关于设备特征的信息应该是什么样的。.
/// 示例实现可能如下所示:
/// ```
/// enum DeviceCharacteristic: String, CharacteristicIdentifier {
///     case manufacturerName = "2A29"
///     var uuid: CBUUID {
///         return CBUUID(string: self.rawValue)
///     }
///     var service: ServiceIdentifier {
///         switch self {
///         case .ManufacturerName:
///             return XXXService.DeviceInformation
///         }
///     }
/// }
/// ```
public protocol CharacteristicIdentifier {
    /// 特征的唯一标识符.
    var uuid: CBUUID { get }
    /// 此特征属于的' ServiceIdentifier '实例.
    var service: ServiceIdentifier { get }
}

/// DescriptorIdentifier 指定了设备描述符的信息应该是什么样的.
public protocol DescriptorIdentifier {
    /// 描述符的唯一标识符
    var uuid: CBUUID { get }
    /// 此描述符所属的`CharacteristicIdentifier`实例
    var characteristic: CharacteristicIdentifier { get }
}
