import Foundation
import CoreBluetooth
import RxSwift

// swiftlint:disable line_length

extension Peripheral {

    /// 接收具有给定标识符的服务。如果缓存中是可用的，则从缓存中取出，否则通过' discoverServices '调用获取
    /// - Parameter identifier: 服务唯一标识符
    /// - Returns: 当服务被发现时`Single` 发出 `next` 事件.
    ///
    /// Observable可以以下列错误结束:
    /// * `RxError.noElements`
    /// * `BluetoothError.servicesDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func service(with identifier: ServiceIdentifier) -> Single<Service> {
        return .deferred { [weak self] in
            guard let strongSelf = self else { throw BluetoothError.destroyed }
            if let services = strongSelf.services,
                let service = services.first(where: { $0.uuid == identifier.uuid }) {
                return .just(service)
            } else {
                return strongSelf.discoverServices([identifier.uuid])
                    .map {
                        if let service = $0.first {
                            return service
                        }
                        throw RxError.noElements
                    }
            }
        }
    }

    /// 接收具有给定标识符的特征。如果缓存中是可用的，则从缓存中取出。否则-直接通过' discoverCharacteristics '调用
    /// - Parameter identifier: 特征的唯一标识符，即具有特征所属的服务信息.
    /// - Returns: 当指定的特征被发现时`Single` 发出 `next` 事件, .
    ///
    /// Observable可以以下列错误结束:
    /// * `RxError.noElements`
    /// * `BluetoothError.characteristicsDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func characteristic(with identifier: CharacteristicIdentifier) -> Single<Characteristic> {
        return .deferred { [weak self] in
            guard let strongSelf = self else { throw BluetoothError.destroyed }
            return strongSelf.service(with: identifier.service)
                .flatMap { service -> Single<Characteristic> in
                    if let characteristics = service.characteristics, let characteristic = characteristics.first(where: {
                        $0.uuid == identifier.uuid
                    }) {
                        return .just(characteristic)
                    }
                    return service.discoverCharacteristics([identifier.uuid])
                        .map {
                            if let characteristic = $0.first {
                                return characteristic
                            }
                            throw RxError.noElements
                        }
                }
        }
    }

    /// 接收具有给定标识符的描述符。如果缓存中是可用的，就从缓存中取出。否则-直接通过' discoverDescriptor '调用
    /// - Parameter identifier: 描述符的唯一标识符，包含描述符所属特征的信息.
    /// - Returns: 当指定的描述符被发现时`Single` 发出 `next` 事件.
    ///
    /// Observable可以以下列错误结束:
    /// * `RxError.noElements`
    /// * `BluetoothError.descriptorsDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func descriptor(with identifier: DescriptorIdentifier) -> Single<Descriptor> {
        return .deferred { [weak self] in
            guard let strongSelf = self else { throw BluetoothError.destroyed }
            return strongSelf.characteristic(with: identifier.characteristic)
                .flatMap { characteristic -> Single<Descriptor> in
                    if let descriptors = characteristic.descriptors,
                        let descriptor = descriptors.first(where: { $0.uuid == identifier.uuid }) {
                        return .just(descriptor)
                    }
                    return characteristic.discoverDescriptors()
                        .map {
                            if let descriptor = $0.filter({ $0.uuid == identifier.uuid }).first {
                                return descriptor
                            }
                            throw RxError.noElements
                        }
                }
        }
    }

    /// 观察特征所发生的写操作的函数.
    /// - Parameter identifier: 应该观察写值特征的标识符.
    /// - Returns: 每当特征完成写入操作时 Observable 发出`next`.
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeWrite(for identifier: CharacteristicIdentifier)
        -> Observable<Characteristic> {
        return characteristic(with: identifier)
            .asObservable()
            .flatMap { [weak self] in
                self?.observeWrite(for: $0) ?? .error(BluetoothError.destroyed)
            }
    }

    /// 将数据写入特征, 在订阅“Observable”之后调用Write.
    /// 这个函数的行为强烈依赖于 [CBCharacteristicWriteType](https://developer.apple.com/documentation/corebluetooth/cbcharacteristicwritetype),
    /// 所以在使用这个方法之前一定要检查一下.
    /// - parameter data: 将被写入“特征”实例的数据
    /// - parameter forCharacteristicWithIdentifier: 特征的唯一标识符，该标识符还包含有关服务特征的信息.
    /// - parameter type: 写操作的类型, 可能的值: `.withResponse`, `.withoutResponse`
    /// - returns: Observable 发送事件依赖于 `CBCharacteristicWriteType` 在函数中的调用.
    /// 行为如下:
    /// - `withResponse` -  确认写入没有任何错误，Observable 将发出 带有 Characteristic 实例的 '.next'事件，接着发送 '.complete'事件。如果发生任何错误则会发出'.error'
    /// - `withoutResponse` - 被调用后，Observable发出带有“Characteristic”实例的“next”. 紧接着调用' .complete '.
    ///此调用的结果未被检查，因此作为用户，您不确定是否一切都已成功完成，不会发出错误。它通过监听适当的委托方法来确保外围设备已经准备好写入而没有响应
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func writeValue(_ data: Data, for identifier: CharacteristicIdentifier,
                           type: CBCharacteristicWriteType) -> Single<Characteristic> {
        return characteristic(with: identifier)
            .flatMap { [weak self] in
                self?.writeValue(data, for: $0, type: type) ?? .error(BluetoothError.destroyed)
            }
    }

    /// 观察特则值更新.
    /// - Parameter identifier: 特征的唯一标识符，也包含特征所属的服务信息.
    /// - Returns: 当值发生改变时 Observable 发出 `next` .
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeValueUpdate(for identifier: CharacteristicIdentifier) -> Observable<Characteristic> {
        return characteristic(with: identifier)
            .asObservable()
            .flatMap { [weak self] in
                self?.observeValueUpdate(for: $0).asObservable() ?? .error(BluetoothError.destroyed)
            }
    }

    /// 读取“特征”实例的当前值， Read是在订阅“Observable”之后调用的
    /// - Parameter identifier: 特征的唯一标识符，也包含特征所属的服务信息.
    /// - Returns: 当特征值准备读取时 Observable 发出 `next` ，  紧接着发出' .complete '.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func readValue(for identifier: CharacteristicIdentifier) -> Single<Characteristic> {
        return characteristic(with: identifier)
            .flatMap { [weak self] in
                self?.readValue(for: $0) ?? .error(BluetoothError.destroyed)
            }
    }

    /// 设置特征通知, 并特征被更改时接收回调.
    /// 返回的可观察者，在指定特征每当值发生改变时发送 .next 事件.
    /// 可以为相同的特征设置更多的可观察对象，并且通知的生命周期将在它们之间共享.
    ///
    /// 一旦这个可观察对象被取消订阅，通知将自动取消注册
    ///
    /// - parameter characteristic: 通知设置的“特征”.
    /// - returns: 每当特征的值发生台变是`Observable` 发出 `Characteristic` 的 .next 事件.
    ///
    /// 这是**无限**的值流.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeValueUpdateAndSetNotification(for identifier: CharacteristicIdentifier)
        -> Observable<Characteristic> {
        return characteristic(with: identifier)
            .asObservable()
            .flatMap { [weak self] in
                self?.observeValueUpdateAndSetNotification(for: $0) ?? .error(BluetoothError.destroyed)
            }
    }

    /// Function that triggers descriptors discovery for characteristic
    /// - parameter identifier: unique identifier of descriptor, which also holds information about characteristic that descriptor belongs to.
    /// - Returns: `Single` that emits `next` with array of `Descriptor` instances, once they're discovered.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorsDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func discoverDescriptors(for identifier: CharacteristicIdentifier) ->
        Single<[Descriptor]> {
        return characteristic(with: identifier)
            .flatMap { [weak self] in
                self?.discoverDescriptors(for: $0) ?? .error(BluetoothError.destroyed)
            }
    }

    /// 观察发生在描述符上的写操作.
    /// - parameter identifier: 描述符的唯一标识符，它还包含描述符所属的特征信息.
    /// - Returns: 描述符每当发生写操作时 Observable 发出 next` 事件 .
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeWrite(for identifier: DescriptorIdentifier) -> Observable<Descriptor> {
        return descriptor(with: identifier)
            .asObservable()
            .flatMap { [weak self] in
                self?.observeWrite(for: $0) ?? .error(BluetoothError.destroyed)
            }
    }

    /// 将数据写入描述符， 在订阅“Observable”之后调用Write.
    /// - parameter data:  Data '将被写入' Descriptor '实例
    /// - parameter identifier: 描述符的唯一标识符，它还包含描述符所属的特征信息.
    /// - returns: 一旦成功写入值，‘Single’将发出‘next’.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func writeValue(_ data: Data, for identifier: DescriptorIdentifier)
        -> Single<Descriptor> {
        return descriptor(with: identifier)
            .flatMap { [weak self] in
                self?.writeValue(data, for: $0) ?? .error(BluetoothError.destroyed)
            }
    }

    /// 观察“描述符”实例的值更新.
    /// - parameter identifier: 描述符的唯一标识符，它还包含描述符所属的特征信息.
    /// - Returns: 当值发生改变时， ‘Single’将发出‘next’
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeValueUpdate(for identifier: DescriptorIdentifier) -> Observable<Descriptor> {
        return descriptor(with: identifier)
            .asObservable()
            .flatMap { [weak self] in
                self?.observeValueUpdate(for: $0) ?? .error(BluetoothError.destroyed)
            }
    }

    /// 读取 Descriptor 当前值， Read是在订阅“Observable”之后调用的
    /// - Parameter identifier: `用来读取值的描述符
    /// - Returns: 当值准备读取时，发出' next '， 紧接着发出' .complete '.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func readValue(for identifier: DescriptorIdentifier) -> Single<Descriptor> {
        return descriptor(with: identifier)
            .flatMap { [weak self] in
                self?.readValue(for: $0) ?? .error(BluetoothError.destroyed)
            }
    }
}
