import Foundation
import RxSwift
import CoreBluetooth


/// Peripheral 是一个实现ReactiveX API的类，它封装了所有核心蓝牙功能，
/// 允许与外设通信，比如发现特征、服务和所有的读/写调用.
public class Peripheral {

    /// 用于蓝牙通信的CentralManager的Intance
    public unowned let manager: CentralManager

    /// CBPeripheral 实例
    public let peripheral: CBPeripheral

    /// 负责Characteristic通知的观察
    private let notificationManager: CharacteristicNotificationManager

    let delegateWrapper: CBPeripheralDelegateWrapper

    private let remainingServicesDiscoveryRequest = ThreadSafeBox<Int>(value: 0)
    private let peripheralDidDiscoverServices = PublishSubject<([CBService]?, Error?)>()

    private let remainingIncludedServicesDiscoveryRequest = ThreadSafeBox<[CBUUID: Int]>(value: [CBUUID: Int]())
    private let peripheralDidDiscoverIncludedServicesForService = PublishSubject<(CBService, Error?)>()

    private let remainingCharacteristicsDiscoveryRequest = ThreadSafeBox<[CBUUID: Int]>(value: [CBUUID: Int]())
    private let peripheralDidDiscoverCharacteristicsForService = PublishSubject<(CBService, Error?)>()

    private let disposeBag = DisposeBag()

    /// 创建新的 `Peripheral`
    /// - parameter manager: 中心实例，用于执行所有必要的操作.
    /// - parameter peripheral: 表示允许在其上执行操作的特定外围设备实例。.
    /// - parameter delegateWrapper: CBPeripheralDelegate 的 ReactiveX 包装.
    /// - parameter notificationManager: 用于观察特征通知的实例
    init(
        manager: CentralManager,
        peripheral: CBPeripheral,
        delegateWrapper: CBPeripheralDelegateWrapper,
        notificationManager: CharacteristicNotificationManager
    ) {
        self.manager = manager
        self.peripheral = peripheral
        self.delegateWrapper = delegateWrapper
        self.notificationManager = notificationManager
        peripheral.delegate = self.delegateWrapper

        setupSubjects()
    }

    convenience init(manager: CentralManager,
                     peripheral: CBPeripheral,
                     delegateWrapper: CBPeripheralDelegateWrapper) {
        let notificationManager = CharacteristicNotificationManager(peripheral: peripheral, delegateWrapper: delegateWrapper)
        self.init(manager: manager,
                  peripheral: peripheral,
                  delegateWrapper: delegateWrapper,
                  notificationManager: notificationManager)
    }

    private func setupSubjects() {
        manager.delegateWrapper
            .didDisconnectPeripheral
            .filter { [weak self] peripheral, _ in
                peripheral.uuidIdentifier == self?.peripheral.uuidIdentifier
            }
            .subscribe(onNext: { [weak self] _ in
                self?.remainingServicesDiscoveryRequest.writeSync { value in
                    value = 0
                }

                self?.remainingIncludedServicesDiscoveryRequest.writeSync { array in
                    array.removeAll()
                }

                self?.remainingCharacteristicsDiscoveryRequest.writeSync { array in
                    array.removeAll()
                }
            })
            .disposed(by: disposeBag)

        delegateWrapper.peripheralDidDiscoverServices.subscribe { [weak self] event in
            self?.remainingServicesDiscoveryRequest.writeSync { value in
                if value > 0 {
                    value -= 1
                }
            }
            self?.peripheralDidDiscoverServices.on(event)
        }.disposed(by: disposeBag)

        delegateWrapper.peripheralDidDiscoverIncludedServicesForService.subscribe { [weak self] event in
            self?.remainingIncludedServicesDiscoveryRequest.writeSync { array in
                if let element = event.element {
                    let oldValue = array[element.0.uuid] ?? 1
                    if oldValue > 0 {
                        array[element.0.uuid] = oldValue - 1
                    }
                }
            }
            self?.peripheralDidDiscoverIncludedServicesForService.on(event)
        }.disposed(by: disposeBag)

        delegateWrapper.peripheralDidDiscoverCharacteristicsForService.subscribe { [weak self] event in
            self?.remainingCharacteristicsDiscoveryRequest.writeSync { array in
                if let element = event.element {
                    let oldValue = array[element.0.uuid] ?? 1
                    if oldValue > 0 {
                        array[element.0.uuid] = oldValue - 1
                    }
                }
            }
            self?.peripheralDidDiscoverCharacteristicsForService.on(event)
        }.disposed(by: disposeBag)
    }

    /// 将BluetoothKit委托连接到CBPeripheral.
    /// 当CBPeripheral的委托被重新分配到BluetoothKit库之外时，这种方法很有用(例如，CBPeripheral被用在其他一些库中，或者以非反应性的方式使用)
    public func attach() {
        peripheral.delegate = delegateWrapper
    }

    /// 指示外设当前是否处于连接状态.
    public var isConnected: Bool {
        return peripheral.state == .connected
    }

    ///  述的“外围设备”实例的当前状态 [CBPeripheralState](https://developer.apple.com/documentation/corebluetooth/cbperipheralstate).
    ///  - returns: 外围设备的当前状态为“CBPeripheralState”`.
    public var state: CBPeripheralState {
        return peripheral.state
    }

    /// “外围设备”实例的当前名称。类似于 [name](https://developer.apple.com/documentation/corebluetooth/cbperipheral/1519029-name) of `CBPeripheral`.
    public var name: String? {
        return peripheral.name
    }

    /// “外围设备”实例的唯一标识符。一旦外围设备被系统发现就被分配.
    public var identifier: UUID {
        return peripheral.uuidIdentifier
    }

    /// 已发现的服务的列表。类似于 [services](https://developer.apple.com/documentation/corebluetooth/cbperipheral/1518978-services) of `CBPeripheral`.
    public var services: [Service]? {
        return peripheral.services?.map {
            Service(peripheral: self, service: $0)
        }
    }

    /// 如果远程设备有空间发送无响应的写，则为YES。
    /// 如果该值为NO，则在当前写操作被刷新后将该值设置为YES，并调用' peripheralIsReadyToSendWriteWithoutResponse: '。
    public var canSendWriteWithoutResponse: Bool {
        if #available(iOS 11.0, macOS 10.13, tvOS 11.0, watchOS 4.0, *) {
            return peripheral.canSendWriteWithoutResponse
        } else {
            return true
        }
    }

    // MARK: 连接

    ///  与外设是否连接的Observable. 当连接状态时发出 .next
    /// - 返回Observable，当“外围设备”连接或断开连接时，它会发出 .next.
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeConnection() -> Observable<Bool> {
        let disconnected = manager.observeDisconnect(for: self).map { _ in false }
        let connected = manager.observeConnect(for: self).map { _ in true }
        return Observable.of(disconnected, connected).merge()
    }

    /// 与给定的“外围设备”建立连接.
    /// 更多信息请查看 `CentralManager.establishConnection(with:options:)` 因为这个方法直接调用它
    /// - parameter options: 字典来定制连接的行为.
    /// - returns: ' Observable '在连接建立后发出' next '事件.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.peripheralIsAlreadyObservingConnection`
    /// * `BluetoothError.peripheralConnectionFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func establishConnection(options: [String: Any]? = nil) -> Observable<Peripheral> {
        return manager.establishConnection(self, options: options)
    }

    // MARK: 服务

    /// 触发器发现指定的外围设备服务。如果servicesUUIDs参数为nil，则返回外围设备的所有可用服务;将参数设置为nil要慢得多，不建议这样做。
    /// 如果已经发现了所有指定的服务，则返回这些服务，而不进行任何底层蓝牙操作.
    ///
    /// - Parameter serviceUUIDs: An array of [CBUUID](https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBUUID_Class/)
    /// 您感兴趣的对象。在这里每一个 [CBUUID](https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBUUID_Class/)
    /// 对象表示一个UUID，该UUID标识希望发现的服务类型.
    /// - Returns: 一旦发现‘服务’实例数组，‘Single’会发出‘next’。
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.servicesDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func discoverServices(_ serviceUUIDs: [CBUUID]?) -> Single<[Service]> {
        if let identifiers = serviceUUIDs, !identifiers.isEmpty,
            let cachedServices = self.services,
            let filteredServices = filterUUIDItems(uuids: serviceUUIDs, items: cachedServices, requireAll: true) {
            return ensureValidPeripheralState(for: .just(filteredServices)).asSingle()
        }
        let observable = peripheralDidDiscoverServices
            .filter { [weak self] (services, error) in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                guard let cachedServices = strongSelf.services, error == nil else { return true }
                let foundRequestedServices = serviceUUIDs != nil && filterUUIDItems(uuids: serviceUUIDs, items: cachedServices, requireAll: true) != nil
                return foundRequestedServices || strongSelf.remainingServicesDiscoveryRequest.read { $0 == 0 }
            }
            .flatMap { [weak self] (_, error) -> Observable<[Service]> in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                guard let cachedServices = strongSelf.services, error == nil else {
                    throw BluetoothError.servicesDiscoveryFailed(strongSelf, error)
                }
                if let filteredServices = filterUUIDItems(uuids: serviceUUIDs, items: cachedServices, requireAll: false) {
                    return .just(filteredServices)
                }
                return .empty()
            }
            .take(1)

        return ensureValidPeripheralStateAndCallIfSucceeded(
            for: observable,
            postSubscriptionCall: { [weak self] in
                self?.remainingServicesDiscoveryRequest.writeSync { value in
                    value += 1
                }
                self?.peripheral.discoverServices(serviceUUIDs)
            }
        )
        .asSingle()
    }

    /// 触发的函数包括指定服务的服务发现。在订阅“Observable”之后调用发现.
    /// 如果已经发现了所有指定的包含服务，则返回这些服务而不执行任何底层蓝牙操作。
    ///
    /// - Parameter includedServiceUUIDs: 应该被发现的包含服务的标识符。如果为“nil”，则会发现所有包含的服务。如果你传递一个空数组——它们都不会被发现.
    /// - Parameter service: 应该发现其中包含的服务.
    /// - Returns: `一旦发现‘服务’实例数组，就会发出‘next’。
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.includedServicesDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func discoverIncludedServices(_ includedServiceUUIDs: [CBUUID]?, for service: Service) -> Single<[Service]> {
        if let identifiers = includedServiceUUIDs, !identifiers.isEmpty,
            let services = service.includedServices,
            let filteredServices = filterUUIDItems(uuids: includedServiceUUIDs, items: services, requireAll: true) {
            return ensureValidPeripheralState(for: .just(filteredServices)).asSingle()
        }
        let observable = peripheralDidDiscoverIncludedServicesForService
            .filter { $0.0 == service.service }
            .filter { [weak self] (cbService, error) in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                guard let includedCBServices = cbService.includedServices, error == nil else { return true }

                let includedServices = includedCBServices.map { Service(peripheral: strongSelf, service: $0) }
                let foundRequestedServices = includedServiceUUIDs != nil && filterUUIDItems(uuids: includedServiceUUIDs, items: includedServices, requireAll: true) != nil
                return foundRequestedServices || strongSelf.remainingIncludedServicesDiscoveryRequest.read { array in
                    return (array[cbService.uuid] ?? 0) == 0
                }
            }
            .flatMap { [weak self] (cbService, error) -> Observable<[Service]> in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                guard let includedRxServices = cbService.includedServices, error == nil else {
                    throw BluetoothError.includedServicesDiscoveryFailed(strongSelf, error)
                }
                let includedServices = includedRxServices.map { Service(peripheral: strongSelf, service: $0) }
                if let filteredServices = filterUUIDItems(uuids: includedServiceUUIDs, items: includedServices, requireAll: false) {
                    return .just(filteredServices)
                }
                return .empty()
            }
            .take(1)

        return ensureValidPeripheralStateAndCallIfSucceeded(
            for: observable,
            postSubscriptionCall: { [weak self] in
                self?.remainingIncludedServicesDiscoveryRequest.writeSync { array in
                    let oldValue = array[service.uuid] ?? 0
                    array[service.uuid] = oldValue + 1
                }
                self?.peripheral.discoverIncludedServices(includedServiceUUIDs, for: service.service)
            }
        )
        .asSingle()
    }

    // MARK: 特征

    /// 该函数为指定的服务和标识符触发特征发现。在订阅“Observable”之后调用发现.
    /// 如果已经发现了所有指定的特征，则返回这些特征，而不进行任何底层蓝牙操作.
    ///
    /// - Parameter characteristicUUIDs: 应该被发现的特征的标识符。如果' nil ' -所有特征将被发现。如果你传递一个空数组——它们都不会被发现.
    /// - Parameter service: 应该发现哪些特征的服务.
    /// - Returns: 一旦被发现，‘Single’会发出带有Characteristic’实例数组的‘next’.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicsDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func discoverCharacteristics(_ characteristicUUIDs: [CBUUID]?, for service: Service) -> Single<[Characteristic]> {
        if let identifiers = characteristicUUIDs, !identifiers.isEmpty,
            let characteristics = service.characteristics,
            let filteredCharacteristics = filterUUIDItems(uuids: characteristicUUIDs, items: characteristics, requireAll: true) {
            return ensureValidPeripheralState(for: .just(filteredCharacteristics)).asSingle()
        }
        let observable = peripheralDidDiscoverCharacteristicsForService
            .filter { $0.0 == service.service }
            .filter { [weak self] (cbService, error) in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                guard let cbCharacteristics = cbService.characteristics, error == nil else { return true }

                let characteristics = cbCharacteristics.map { Characteristic(characteristic: $0, service: service) }
                let foundRequestedCharacteristis = characteristicUUIDs != nil && filterUUIDItems(uuids: characteristicUUIDs, items: characteristics, requireAll: true) != nil
                return foundRequestedCharacteristis || strongSelf.remainingCharacteristicsDiscoveryRequest.read { array in
                    return (array[cbService.uuid] ?? 0) == 0
                }
            }
            .flatMap { (cbService, error) -> Observable<[Characteristic]> in
                guard let cbCharacteristics = cbService.characteristics, error == nil else {
                    throw BluetoothError.characteristicsDiscoveryFailed(service, error)
                }
                let characteristics = cbCharacteristics.map { Characteristic(characteristic: $0, service: service) }
                if let filteredCharacteristics = filterUUIDItems(uuids: characteristicUUIDs, items: characteristics, requireAll: false) {
                    return .just(filteredCharacteristics)
                }
                return .empty()
            }
            .take(1)

        return ensureValidPeripheralStateAndCallIfSucceeded(
            for: observable,
            postSubscriptionCall: { [weak self] in
                self?.remainingCharacteristicsDiscoveryRequest.writeSync { array in
                    let oldValue = array[service.uuid] ?? 0
                    array[service.uuid] = oldValue + 1
                }
                self?.peripheral.discoverCharacteristics(characteristicUUIDs, for: service.service)
            }
        ).asSingle()
    }

    /// 允许观察特征所发生的写操作的函数.
    /// - Parameter characteristic: 可选的“特征”，其值的变化应该被观察到。当未指定时，它将观察任何“特性”.
    /// - Returns: 每当发生写操作时Observable将发出带有Characteristic实例的`next`事件.
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeWrite(for characteristic: Characteristic? = nil) -> Observable<Characteristic> {
        let observable = delegateWrapper
            .peripheralDidWriteValueForCharacteristic
            .filter { characteristic != nil ? ($0.0 == characteristic!.characteristic) : true }
            .map { [weak self] (cbCharacteristic, error) -> Characteristic in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                let characteristic = characteristic ?? Characteristic(characteristic: cbCharacteristic, peripheral: strongSelf)
                if let error = error {
                    throw BluetoothError.characteristicWriteFailed(characteristic, error)
                }
                return characteristic
            }
        return ensureValidPeripheralState(for: observable)
    }

    /// 在一次写操作中可以发送给一个特征的最大数据量，以字节为单位.
    /// - parameter type: 写操作的类型。可能的值: `.withResponse`, `.withoutResponse`
    /// - seealso: `writeValue(_:for:type:)`
    @available(OSX 10.12, iOS 9.0, *)
    public func maximumWriteValueLength(for type: CBCharacteristicWriteType) -> Int {
        return peripheral.maximumWriteValueLength(for: type)
    }

    /// 触发将数据写入特征的函数。在订阅“Observable”之后调用Write.
    /// 这个函数的行为强烈依赖于 [CBCharacteristicWriteType](https://developer.apple.com/documentation/corebluetooth/cbcharacteristicwritetype),
    /// 所以在使用这个方法之前一定要检查一下.
    ///
    /// 行为如下:
    /// - `withResponse` -  确认写入没有任何错误，Observable 将发出 带有 Characteristic 实例的 '.next'事件，接着发送 '.complete'事件。如果发生任何错误则会发出'.error'
    /// - `withoutResponse` - 被调用后，Observable发出带有“Characteristic”实例的“next”. 紧接着调用' .complete '.
    ///此调用的结果未被检查，因此作为用户，您不确定是否一切都已成功完成，不会发出错误。它通过监听适当的委托方法来确保外围设备已经准备好写入而没有响应
    ///
    /// - parameter data: 将被写入“特征”实例的数据
    /// - parameter characteristic: 被写入数据的特征实例.
    /// - parameter type: 写操作的类型。可能的值: `.withResponse`, `.withoutResponse`
    /// - parameter canSendWriteWithoutResponseCheckEnabled: 检查是否应该启用canSendWriteWithoutResponse。因为MacOS内部的bug.
    /// - returns: Observable 发送事件依赖于 `CBCharacteristicWriteType` 在函数中的调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func writeValue(_ data: Data,
                           for characteristic: Characteristic,
                           type: CBCharacteristicWriteType,
                           canSendWriteWithoutResponseCheckEnabled: Bool = true) -> Single<Characteristic> {
        let writeOperationPerformingAndListeningObservable = { [weak self] (observable: Observable<Characteristic>)
            -> Observable<Characteristic> in
            guard let strongSelf = self else { return Observable.error(BluetoothError.destroyed) }
            return strongSelf.ensureValidPeripheralStateAndCallIfSucceeded(
                for: observable,
                postSubscriptionCall: { [weak self] in
                    self?.peripheral.writeValue(data, for: characteristic.characteristic, type: type)
                }
            )
        }
        switch type {
        case .withoutResponse:
            return observeWriteWithoutResponseReadiness()
                .map { _ in true }
                .startWith(canSendWriteWithoutResponseCheckEnabled ? canSendWriteWithoutResponse : true)
                .filter { $0 }
                .take(1)
                .flatMap { _ in
                    writeOperationPerformingAndListeningObservable(Observable.just(characteristic))
                }.asSingle()
        case .withResponse:
            return writeOperationPerformingAndListeningObservable(observeWrite(for: characteristic).take(1))
                .asSingle()
        @unknown default:
            return .error(BluetoothError.unknownWriteType)
        }
    }

    /// 允许观察“特征”实例的值更新的函数.
    /// - Parameter characteristic: 可选的“特征”，其值的变化应该被观察到。当未指定时，它将观察任何“特性”.
    /// - Returns: 被观察到的，在每次值改变时发出' next '和' Characteristic '实例.
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeValueUpdate(for characteristic: Characteristic? = nil) -> Observable<Characteristic> {
        let observable = delegateWrapper
            .peripheralDidUpdateValueForCharacteristic
            .filter { characteristic != nil ? ($0.0 == characteristic!.characteristic) : true }
            .map { [weak self] (cbCharacteristic, error) -> Characteristic in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                let characteristic = characteristic ?? Characteristic(characteristic: cbCharacteristic, peripheral: strongSelf)
                if let error = error {
                    throw BluetoothError.characteristicReadFailed(characteristic, error)
                }
                return characteristic
            }
        return ensureValidPeripheralState(for: observable)
    }

    /// 该函数触发读取“特征”实例的当前值。Read是在订阅“Observable”之后调用的.
    /// - Parameter characteristic: 要读取值的`Characteristic`
    /// - Returns: 当值准备读取时`Single`发出带有characteristic的 `next` .
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func readValue(for characteristic: Characteristic) -> Single<Characteristic> {
        let observable = observeValueUpdate(for: characteristic).take(1)
        return ensureValidPeripheralStateAndCallIfSucceeded(
            for: observable,
            postSubscriptionCall: { [weak self] in
                self?.peripheral.readValue(for: characteristic.characteristic)
            }
        ).asSingle()
    }

    /// 设置特征通知，以便在给定特征被更改时接收回调.
    /// 返回的被观察对象将在每个通知变化上发出“特征”.
    /// 可以为相同的特征设置更多的可观察对象，并且通知的生命周期将在它们之间共享.
    ///
    /// 一旦这个可观察对象被取消订阅，通知将自动取消注册
    ///
    /// - parameter characteristic: 通知设置的“特征”。
    /// - returns: 当给定的特性改变时`Observable` 发出 `Characteristic`.
    ///
    /// 这是**无限**的值流.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeValueUpdateAndSetNotification(for characteristic: Characteristic) -> Observable<Characteristic> {
        let observable = notificationManager.observeValueUpdateAndSetNotification(for: characteristic)
        return ensureValidPeripheralState(for: observable)
    }

    /// 使用这个函数可以知道某个特性的isNotyfing值发生变化的确切时间.
    ///
    /// - parameter characteristic:要观察 isNotyfing值改变的`Characteristic` .
    /// - returns: 当给定特征的isNotyfing值发生改变时， Observable 发出 .next .
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicSetNotifyValueFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeNotifyValue(for characteristic: Characteristic) -> Observable<Characteristic> {
        return delegateWrapper.peripheralDidUpdateNotificationStateForCharacteristic
            .filter { $0.0 == characteristic.characteristic }
            .map { [weak self] (cbCharacteristic, error) -> Characteristic in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                let characteristic = Characteristic(characteristic: cbCharacteristic, peripheral: strongSelf)
                if let error = error {
                    throw BluetoothError.characteristicSetNotifyValueFailed(characteristic, error)
                }
                return characteristic
        }
    }

    // MARK: 描述符

    /// 触发特征描述符发现的函数
    /// 如果已经发现了所有的描述符—返回这些描述符时不进行任何底层蓝牙操作。
    /// - Parameter characteristic: 需要发现Descriptor指定的特征.
    /// - Returns: 一旦描述符被发现， Single 将发出 .next .
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorsDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func discoverDescriptors(for characteristic: Characteristic) -> Single<[Descriptor]> {
        if let descriptors = characteristic.descriptors {
            let resultDescriptors = descriptors.map { Descriptor(descriptor: $0.descriptor, characteristic: characteristic) }
            return ensureValidPeripheralState(for: .just(resultDescriptors)).asSingle()
        }
        let observable = delegateWrapper
            .peripheralDidDiscoverDescriptorsForCharacteristic
            .filter { $0.0 == characteristic.characteristic }
            .take(1)
            .map { (cbCharacteristic, error) -> [Descriptor] in
                if let descriptors = cbCharacteristic.descriptors, error == nil {
                    return descriptors.map {
                        Descriptor(descriptor: $0, characteristic: characteristic) }
                }
                throw BluetoothError.descriptorsDiscoveryFailed(characteristic, error)
            }

        return ensureValidPeripheralStateAndCallIfSucceeded(
            for: observable,
            postSubscriptionCall: { [weak self] in
                self?.peripheral.discoverDescriptors(for: characteristic.characteristic)
            }
        ).asSingle()
    }

    /// 函数，该函数允许观察发生在描述符上的写操作.
    /// - Parameter descriptor: 可选的“描述符”，写入操作被观察。当未指定时，它将观察任何“描述符”.
    /// - Returns: 当写操作发生时，会发出带有“描述符”实例的“next”.
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeWrite(for descriptor: Descriptor? = nil) -> Observable<Descriptor> {
        let observable = delegateWrapper
            .peripheralDidWriteValueForDescriptor
            .filter { descriptor != nil ? ($0.0 == descriptor!.descriptor) : true }
            .map { [weak self] (cbDescriptor, error) -> Descriptor in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                let descriptor = descriptor ?? Descriptor(descriptor: cbDescriptor, peripheral: strongSelf)
                if let error = error {
                    throw BluetoothError.descriptorWriteFailed(descriptor, error)
                }
                return descriptor
            }
        return ensureValidPeripheralState(for: observable)
    }

    /// 函数，允许观察“描述符”实例的值更新.
    /// - Parameter descriptor: 可选的“描述符”，值的变化应该被观察。当未指定时，它将观察任何“描述符”.
    /// - Returns: Observable that emits `next` with `Descriptor` instance every time when value has changed.
    /// 它是**infinite**流，所以' .complete '不会被调用.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeValueUpdate(for descriptor: Descriptor? = nil) -> Observable<Descriptor> {
        let observable = delegateWrapper
            .peripheralDidUpdateValueForDescriptor
            .filter { descriptor != nil ? ($0.0 == descriptor!.descriptor) : true }
            .map { [weak self] (cbDescriptor, error) -> Descriptor in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                let descriptor = descriptor ?? Descriptor(descriptor: cbDescriptor, peripheral: strongSelf)
                if let error = error {
                    throw BluetoothError.descriptorReadFailed(descriptor, error)
                }
                return descriptor
            }
        return ensureValidPeripheralState(for: observable)
    }

    /// 函数，该函数触发读取“描述符”实例的当前值.
    /// Read是在订阅“Observable”之后调用的.
    /// - Parameter descriptor: 从`Descriptor`读取值a
    /// - Returns: 当值准备读取时' Single '发出带有给定描述符的' next '.
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func readValue(for descriptor: Descriptor) -> Single<Descriptor> {
        let observable = observeValueUpdate(for: descriptor).take(1)
        return ensureValidPeripheralStateAndCallIfSucceeded(
            for: observable,
            postSubscriptionCall: { [weak self] in
                self?.peripheral.readValue(for: descriptor.descriptor) }
        )
        .asSingle()
    }

    /// 该函数触发将数据写入描述符。在订阅“Observable”之后调用Write.
    /// - Parameter data: ' Data '将被写入' Descriptor '实例
    /// - Parameter descriptor: 要写入值的“描述符”实例
    /// - Returns: 一旦成功写入值，' Single '将发出' Descriptor '实例的' next '。
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.descriptorWriteFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func writeValue(_ data: Data, for descriptor: Descriptor) -> Single<Descriptor> {
        let observeWrite = self.observeWrite(for: descriptor).take(1)
        return ensureValidPeripheralStateAndCallIfSucceeded(
            for: observeWrite,
            postSubscriptionCall: { [weak self] in
                self?.peripheral.writeValue(data, for: descriptor.descriptor) }
        )
        .asSingle()
    }

    // MARK: 其他方法

    /// 触发读取'外设' RSSI值的函数。Read是在订阅“Observable”之后调用的.
    /// - returns: 一旦读取新的RSSI值，' Single '将发出tuple: `(Peripheral, Int)`
    /// `Int '是新的RSSI值，' Peripheral '被返回以允许更简单的链接
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.peripheralRSSIReadFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func readRSSI() -> Single<(Peripheral, Int)> {
        let observable = delegateWrapper
            .peripheralDidReadRSSI
            .take(1)
            .map { [weak self] (rssi, error) -> (Peripheral, Int) in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                if let error = error {
                    throw BluetoothError.peripheralRSSIReadFailed(strongSelf, error)
                }
                return (strongSelf, rssi)
        }

        return ensureValidPeripheralStateAndCallIfSucceeded(
            for: observable,
            postSubscriptionCall: { [weak self] in
                self?.peripheral.readRSSI()
            }
            ).asSingle()
    }

    /// 观察Peripheral 名称 name 属性的更改.
    /// - returns: 当名称发生更改时`Observable` 发出元组: `(Peripheral, String?)`
    ///    它是“可选字符串”，因为外围设备也可能丢失他的名字.
    ///   它是**无限**的值流，所以' .complete '永远不会被发出。
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeNameUpdate() -> Observable<(Peripheral, String?)> {
        let observable = delegateWrapper.peripheralDidUpdateName.map { [weak self] name -> (Peripheral, String?) in
            guard let strongSelf = self else { throw BluetoothError.destroyed }
            return (strongSelf, name)
        }
        return ensureValidPeripheralState(for: observable)
    }

    /// 观察察“Peripheral”实例的内部服务修改.
    /// 如果您对可能发生的确切变化感兴趣，请参考
    /// [Apple Documentation](https://developer.apple.com/documentation/corebluetooth/cbperipheraldelegate/1518865-peripheral)
    ///
    /// - returns: 当服务被修改时`Observable` 发出元组: `(Peripheral, [Service])` .
    ///   它是**无限**的值流，所以' .complete '永远不会被发出。
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func observeServicesModification() -> Observable<(Peripheral, [Service])> {
        let observable = delegateWrapper.peripheralDidModifyServices
            .map { [weak self] services -> [Service] in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                return services.map { Service(peripheral: strongSelf, service: $0) } }
            .map { [weak self] services -> (Peripheral, [Service]) in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                return (strongSelf, services)
        }
        return ensureValidPeripheralState(for: observable)
    }

    /// 如果调用 `writeValue:forCharacteristic:type:` 失败，Obervable 发出 .next,
    /// 指示外设何时准备再次发送特征值更新.
    public func observeWriteWithoutResponseReadiness() -> Observable<Void> {
        return delegateWrapper.peripheralIsReadyToSendWriteWithoutResponse.asObservable()
    }

    /// 函数 允许`Peripheral` 实例打开 L2CAP 通道 .
    /// 更多信息，请参考
    /// [What’s New in CoreBluetooth, 712, WWDC 2017](https://developer.apple.com/videos/play/wwdc2017/712/)
    ///
    /// - parameter PSM: `PSM` (Protocol/Service Multiplexer) 通道
    /// - returns: 当通道被打开时`Single` 发出 `CBL2CAPChannel`
    /// - since: iOS 11, tvOS 11, watchOS 4
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.openingL2CAPChannelFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    @available(iOS 11, macOS 10.14, tvOS 11, watchOS 4, *)
    public func openL2CAPChannel(PSM: CBL2CAPPSM) -> Single<CBL2CAPChannel> {
        let observable = delegateWrapper
            .peripheralDidOpenL2CAPChannel
            .map {($0.0 as? CBL2CAPChannel, $0.1)}
            .take(1)
            .flatMap { [weak self] (channel, error) -> Observable<CBL2CAPChannel> in
                guard let strongSelf = self else { throw BluetoothError.destroyed }
                if let channel = channel, error == nil {
                    return .just(channel)
                } else {
                    throw BluetoothError.openingL2CAPChannelFailed(strongSelf, error)
                }
        }

        return ensureValidPeripheralStateAndCallIfSucceeded(for: observable, postSubscriptionCall: { [weak self] in
            self?.peripheral.openL2CAPChannel(PSM)
        }).asSingle()
    }

    // MARK: 内部函数

    func ensureValidPeripheralStateAndCallIfSucceeded<T>(for observable: Observable<T>,
                                                         postSubscriptionCall call: @escaping () -> Void
    ) -> Observable<T> {
        let operation = Observable<T>.deferred {
            call()
            return .empty()
        }
        return ensureValidPeripheralState(for: Observable.merge([observable, operation]))
    }

    /// 函数将给定的观察对象与无效的中央管理器状态的错误流合并.
    /// - parameter observable: 将被合并的可观察对象
    /// - returns:  源“可观察到的”，它也监听状态变化错误
    func ensureValidPeripheralState<T>(for observable: Observable<T>) -> Observable<T> {
        return Observable<T>.absorb(
            manager.ensurePeripheralIsConnected(self),
            manager.ensure(.poweredOn, observable: observable)
        )
    }
}

extension Peripheral: Equatable {}

/// 比较标识符相等时相同的两个外围设备。.
///
/// - parameter lhs: 第一个外围
/// - parameter rhs: 第二个外围
/// - returns: 如果两个外设相同，则为真
public func == (lhs: Peripheral, rhs: Peripheral) -> Bool {
    return lhs.peripheral == rhs.peripheral
}
