import Foundation
import CoreBluetooth

/// 类，用于提供外围设备和外围包装器
class PeripheralProvider {

    private let peripheralsBox: ThreadSafeBox<[Peripheral]> = ThreadSafeBox(value: [])

    private let delegateWrappersBox: ThreadSafeBox<[UUID: CBPeripheralDelegateWrapper]> = ThreadSafeBox(value: [:])

    /// 为指定的“CBPeripheral”提供“CBPeripheralDelegateWrapper”
    ///
    /// 如果它是以前创建的，它将返回该对象，因此每个“CBPeripheral”只能有一个“CBPeripheralDelegateWrapper”。
    ///
    /// 如果没有，就创建一个新的.
    ///
    /// - parameter peripheral: 要为其提供委托包装器的外围设备
    /// - returns: 委托指定外围设备的包装器.
    func provideDelegateWrapper(for peripheral: CBPeripheral) -> CBPeripheralDelegateWrapper {
        if let delegateWrapper = delegateWrappersBox.read({ $0[peripheral.uuidIdentifier] }) {
            return delegateWrapper
        } else {
            delegateWrappersBox.compareAndSet(
                compare: { $0[peripheral.uuidIdentifier] == nil },
                set: { $0[peripheral.uuidIdentifier] = CBPeripheralDelegateWrapper()}
            )
            return delegateWrappersBox.read({ $0[peripheral.uuidIdentifier]! })
        }
    }

    /// 为指定的 `CBPeripheral`提供`Peripheral`对象.
    ///
    /// 如果它是以前创建的，它将返回该对象，因此每个“CBPeripheral”只能有一个“Peripheral”。如果没有，就创建一个新的
    ///
    /// - parameter peripheral: 要为其提供委托包装器的外围设备
    /// - returns: 为指定的外围提供的`Peripheral` .
    func provide(for cbPeripheral: CBPeripheral, centralManager: CentralManager) -> Peripheral {
        if let peripheral = find(cbPeripheral) {
            return peripheral
        } else {
            return createAndAddToBox(cbPeripheral, manager: centralManager)
        }
    }

    private  func createAndAddToBox(_ cbPeripheral: CBPeripheral, manager: CentralManager) -> Peripheral {
        peripheralsBox.compareAndSet(
            compare: { peripherals in
                return !peripherals.contains(where: { $0.peripheral == cbPeripheral })
            },
            set: { [weak self] peripherals in
                guard let strongSelf = self else { return }
                let delegateWrapper = strongSelf.provideDelegateWrapper(for: cbPeripheral)
                let newPeripheral = Peripheral(
                    manager: manager,
                    peripheral: cbPeripheral,
                    delegateWrapper: delegateWrapper
                )
                peripherals.append(newPeripheral)
            }
        )
        return peripheralsBox.read { peripherals in
            return peripherals.first(where: { $0.peripheral == cbPeripheral })!
        }
    }

    private  func find(_ cbPeripheral: CBPeripheral) -> Peripheral? {
        return peripheralsBox.read { peripherals in
            return peripherals.first(where: { $0.peripheral == cbPeripheral})
        }
    }
}
