import Foundation
import CoreBluetooth
import RxSwift

/// Service是一个实现ReactiveX的类，它封装了与交互相关的CoreBluetooth功能 [CBService](https://developer.apple.com/library/ios/documentation/CoreBluetooth/Reference/CBService_Class/)
public class Service {
    /// CBService 实例
    public let service: CBService

    /// 此服务所属的外围设备
    public let peripheral: Peripheral

    /// 如果服务是主要服务，则为真
    public var isPrimary: Bool {
        return service.isPrimary
    }

    /// 服务的UUID
    public var uuid: CBUUID {
        return service.uuid
    }

    /// 服务包括服务
    public var includedServices: [Service]? {
        return service.includedServices?.map {
            Service(peripheral: peripheral, service: $0)
        }
    }

    /// 服务的characteristics
    public var characteristics: [Characteristic]? {
        return service.characteristics?.map {
            Characteristic(characteristic: $0, service: self)
        }
    }

    init(peripheral: Peripheral, service: CBService) {
        self.service = service
        self.peripheral = peripheral
    }

    /// 该函数为指定的服务和标识符触发特征发现。在订阅“Observable”之后调用发现。.
    /// - Parameter identifiers: 应该被发现的特征的标识符。如果' nil ' -所有特征将被发现。如果你传递一个空数组——它们都不会被发现.
    /// - Returns: `Single` 发送给定Characteristic数组的 `next` 事件, 一旦他们发现
    /// 如果没有发现所有请求的特征，'RxError.noElements .我们不能排除任何错误
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.characteristicsDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func discoverCharacteristics(_ characteristicUUIDs: [CBUUID]?) -> Single<[Characteristic]> {
        return peripheral.discoverCharacteristics(characteristicUUIDs, for: self)
    }

    /// 触发的函数包括指定服务的服务发现。在订阅“Observable”之后调用发现.
    /// - Parameter includedServiceUUIDs: 应该被发现的包含服务的标识符。如果为“nil”，则会发现所有包含的服务。如果你传递一个空数组——它们都不会被发现。
    /// - Returns: `Single` 将发送 Service 数组实例的`next` 事件, 一旦发现他们.
    /// 如果没有发现所有请求的特征，'RxError.noElements .我们不能排除任何错误
    ///
    /// Observable可以以下列错误结束:
    /// * `BluetoothError.includedServicesDiscoveryFailed`
    /// * `BluetoothError.peripheralDisconnected`
    /// * `BluetoothError.destroyed`
    /// * `BluetoothError.bluetoothUnsupported`
    /// * `BluetoothError.bluetoothUnauthorized`
    /// * `BluetoothError.bluetoothPoweredOff`
    /// * `BluetoothError.bluetoothInUnknownState`
    /// * `BluetoothError.bluetoothResetting`
    public func discoverIncludedServices(_ includedServiceUUIDs: [CBUUID]?) -> Single<[Service]> {
        return peripheral.discoverIncludedServices(includedServiceUUIDs, for: self)
    }
}

extension Service: Equatable {}
extension Service: UUIDIdentifiable {}

/// 比较服务是否相等。如果他们的uuid相同，他们就是.
/// - parameter lhs: 第一个服务
/// - parameter rhs: 第二个服务
/// - returns: 如果服务是相同的，则为真.
public func == (lhs: Service, rhs: Service) -> Bool {
    return lhs.service == rhs.service
}
